/* PhyloDataReader actor reads and parses the data stored in Nexus format.

 Copyright (c) 1998-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
*/

package org.cipres.kepler;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.actor.IOPort;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.Token;
import ptolemy.data.ObjectToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.Type;
import ptolemy.data.expr.Parameter;

import org.cipres.CipresIDL.Rid3TreeImprove;
import org.cipres.CipresIDL.Tree;
import org.cipres.communication.Facilitator;
import org.cipres.datatypes.PhyloDataset;
import org.cipres.datatypes.TreeWrapper;
import org.cipres.helpers.CipresRegistry;
import org.cipres.helpers.CipresServiceDialog;
import org.cipres.helpers.RegistryEntryWrapper;
import org.cipres.util.Config;

//////////////////////////////////////////////////////////////////////////
//// PhyloDataReader
/**
 * This is the implementation of the PhyloDataReader actor using Kepler.
 * This actor reads and parses the data stored in Nexus format.
 * 
 * @author Zhijie Guan  
 * guan@sdsc.edu
*/

public class PhyloDataReader extends TypedAtomicActor {

	/** Construct PhyloDataReader source with the given container and name.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the entity cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
	
    public PhyloDataReader(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);
        
		inputNexusData = new TypedIOPort(this, "Nexus Data Content", true, false);
		inputNexusData.setTypeEquals(BaseType.GENERAL);
		
        outputDataMatrix = new TypedIOPort(this, "Data Matrix", false, true);
        // Set the type constraint.
        outputDataMatrix.setTypeEquals(BaseType.GENERAL);

        outputTree = new TypedIOPort(this, "Tree", false, true);
        // Set the type constraint.
        outputTree.setTypeEquals(BaseType.GENERAL);

        outputTaxaInfo = new TypedIOPort(this, "Taxa Info", false, true);
        // Set the type constraint.
        outputTaxaInfo.setTypeEquals(BaseType.GENERAL);
		
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"20\" "
                + "style=\"fill:white\"/>\n" +
                "</svg>\n");
		
    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** The output port.  The type of this port will be set to String.
     */
	public TypedIOPort inputNexusData = null;
    public TypedIOPort outputTree = null;
	public TypedIOPort outputDataMatrix = null;
	public TypedIOPort outputTaxaInfo = null;
	
	///////////////////////////////////////////////////////////////////
	////                    functional variables                   ////
	private org.cipres.CipresIDL.Tree tree = null;
	private org.cipres.CipresIDL.DataMatrix dataMatrix = null;
	private String[] taxaInfo = null;
	private org.cipres.datatypes.PhyloDataset phyloDataset;
	
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Read and parse the Nexus data.
     *  @exception IllegalActionException If it is thrown by the
     *   send() method sending out the token.
     */
    public void fire() throws IllegalActionException {
        super.fire();
		
		if (inputNexusData.hasToken(0)) {	
			
			try {
				phyloDataset = new PhyloDataset();
				phyloDataset.initialize( ((StringToken)inputNexusData.get(0)).stringValue() );
				
				// get the phylo data
				tree = phyloDataset.getFirstTree();				
				dataMatrix = phyloDataset.getDataMatrix();
				taxaInfo = phyloDataset.getTaxaInfo();
			} catch (Exception ex) {
				System.out.println("Exception on read the phylo data set: ");
				ex.printStackTrace();
			}
			
			// generate the object tokens
			ObjectToken treeToken = new ObjectToken(tree);
			outputTree.send(0, treeToken);
			
			ObjectToken dataMatrixToken = new ObjectToken(dataMatrix);
			outputDataMatrix.send(0, dataMatrixToken);
			
			ObjectToken taxaInfoToken = new ObjectToken(taxaInfo);
			outputTaxaInfo.send(0, taxaInfoToken);
		}
		
    }
	
	/** Post fire the actor. Return false to indicated that the
	 *  process has finished. If it returns true, the process will
	 *  continue indefinitely.
	 */
	public boolean postfire() {
		return false;
	}	
}

