/* TreeImprover actor improves a tree according to the settings configured 
   by the user through the GuiGen interface.

 Copyright (c) 1998-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
*/

package org.cipres.kepler;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.actor.IOPort;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.Token;
import ptolemy.data.ObjectToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.Type;
import ptolemy.data.expr.Parameter;

import javax.swing.JOptionPane;

import org.cipres.CipresIDL.TreeImprove;
import org.cipres.CipresIDL.Tree;
import org.cipres.CipresIDL.TreeScore;
import org.cipres.CipresIDL.DataMatrix;
import org.cipres.helpers.CipresRegistry;
import org.cipres.helpers.RegistryEntryWrapper;
import org.cipres.helpers.CipresServiceDialog;


//////////////////////////////////////////////////////////////////////////
//// TreeImprover
/**
 * This is the implementation of the TreeImprover actor using Kepler.
 * This actor improves a tree according to user's settings.
 * 
 * @author Zhijie Guan  
 * guan@sdsc.edu
*/

public class TreeImprover extends TypedAtomicActor {

	/** Construct TreeImprover source with the given container and name.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the entity cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
	
    public TreeImprover(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);
        
		inputTree = new TypedIOPort(this, "Tree Input", true, false);
		inputTree.setTypeEquals(BaseType.GENERAL);
		inputMatrix = new TypedIOPort(this, "Matrix Input", true, false);
		inputMatrix.setTypeEquals(BaseType.GENERAL);
		
        outputTree = new TypedIOPort(this, "Tree Output", false, true);
        // Set the type constraint.
        outputTree.setTypeEquals(BaseType.GENERAL);
        
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"20\" "
                + "style=\"fill:white\"/>\n" +
                "</svg>\n");
		
    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////
	public TypedIOPort inputTree = null;
	public TypedIOPort inputMatrix = null;
    public TypedIOPort outputTree = null;
	
	///////////////////////////////////////////////////////////////////
	////                    functional variables                   ////
	private org.cipres.CipresIDL.Tree finalTree = null;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Improve a tree and send the final tree to the output port.
     *  @exception IllegalActionException If it is thrown by the
     *   send() method sending out the token.
     */
    public void fire() throws IllegalActionException {
        super.fire();
		if ( (inputTree.hasToken(0)) && (inputMatrix.hasToken(0)) ) {		
			
			RegistryEntryWrapper treeImproverWrapper = null;
			try {
				treeImproverWrapper = CipresRegistry.getCipresServiceWrapper(TreeImprove.class, null, null);
				// with GUI version
				CipresServiceDialog dialog = treeImproverWrapper.getServiceDialog(null);
				int status = dialog.showAndInitialize();
				if (status == CipresServiceDialog.OK)
				{
					TreeImprove service = (TreeImprove)treeImproverWrapper.getService();
					service.setTree( (Tree)((ObjectToken)inputTree.get(0)).getValue() );
					service.setMatrix( (DataMatrix)((ObjectToken)inputMatrix.get(0)).getValue() );
					finalTree = service.improveTree(null);
					
					// fix finalTree. This section should be replaced once the fix_tree is open to public access
					finalTree.m_newick = finalTree.m_newick.trim();
					if (finalTree.m_newick.lastIndexOf(';') == -1)
					{
						finalTree.m_newick += ";";
					}
					if (finalTree.m_score == null)
					{
						finalTree.m_score = new TreeScore();
						finalTree.m_score.noScore(0);
					}
				
				} else if (status == CipresServiceDialog.ERROR)
				{
					JOptionPane.showMessageDialog(null, "error initializing service");
				}
			}
			catch (Exception e) {
				e.printStackTrace();
			}
			finally {
				if (treeImproverWrapper != null) {
					treeImproverWrapper.releaseService();
				}
			}
			
			outputTree.send(0, new ObjectToken( finalTree ));
		}				
    }
}

