/**
 *    '$RCSfile: GARPSummary.java,v $'
 *
 *     '$Author: higgins $'
 *       '$Date: 2006/03/21 20:37:53 $'
 *   '$Revision: 1.4 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.gis.java_gis;

import java.awt.geom.Point2D;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.StringTokenizer;
import java.util.Vector;

import ptolemy.actor.TypedIOPort;
import ptolemy.actor.lib.Transformer;
import ptolemy.data.DoubleToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 *  This code is for examination of the values in individual pixels in
 *  an ascii grid file and summarizing those values
 
 * There are three inputs; 'input'is an ASCI grid (output of GARP); 'pointFileName' is the file of
 * testing locations (long, lat) used to evaluate the prediction; and 'ruleSetFileName is the ruleSetFile
 * which is needed later to reproduce the predicted distribution.<br>
 * The output is a string containing the omission, commission, and the ruleSetFileName, separated
 * by tabs. These should be saved (in a File?) for determination of the 'best' result.<br>
 *
 * Ricardo Periera, provided the following recipe for calculating omission and commision in an
 * e-mail to Dan Higgins, 10/5/2004<br>
 
 * However, those error statistics (omission and commission) could be calculated outside GARP code. 
 * Here is the recipe:<br>
 
 * 1) Get a set of presence data points (species occurrences - x, y coordinates) to test<br>
 * 2) Project the GARP model onto geography (map generated by GarpProjection actor)<br>
 * 3) Overlay the presence points from item #1 onto the map generated on #2. The percentage of those 
 *  points that fall in a pixed not predicted present is your OMISSION. Say, out of 100 points, only 
 *  45 fall on white pixels, the other 55 fall on black ones, your omission is 55% or 0.55.<br>
 * 4) Commission, when we don't have real absence points (our case) is the proportion of area predicted 
 *  present with regard to the total area of interest, not counting masked pixels. So if 40% of the area 
 *  is predicted present, your commission error is 40% or 0.40.<br>
 * 5) Then, select those GARP runs that show omission below a certain omission threshold, say 5 or 10%.<br>
 * 6) From those runs selected in #5, sort them by commission error, and then get the 50% of the models 
 *  that are around the median value for commission. If you got, say, 20 models in item #5, now you have 
 *  10 models that make up your best subset of models.<br>
 * 7) Sum up the maps for the best subset of models in item #6, that is your final prediction map for your 
 *  species.<br>
 * @author Dan Higgins NCEAS UC Santa Barbara
 */
public class GARPSummary extends Transformer {

  //input ports
 /**
  *'pointFileName' is the file of
  * testing locations (long, lat) used to evaluate the prediction
  */
  public TypedIOPort pointFileName;
 /**
  * 'ruleSetFileName is the ruleSetFile
  * which is needed later to reproduce the predicted distribution.
  */
  public TypedIOPort ruleSetFileName;
  
 /**
  * The omission value
  */
  public TypedIOPort omissionValue;
 /**
  * The commission value
  */
  public TypedIOPort commissionValue;
 /**
  * output ruleset file name
  */
  public TypedIOPort outputRuleSetFileName;

  Vector pointList;  // vector of points as Point2D.Double objects  
  
  private double hit_value = 1.0E0; // indicated occurence in GARP output
  private double miss_distance = 1.0e-3;
  private Grid inputGrid;

 
  /**
   * constructor
   *
   *@param  container                     The container.
   *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public GARPSummary(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);
    pointFileName = new TypedIOPort(this, "pointFileName", true, false);
    pointFileName.setTypeEquals(BaseType.STRING);
    ruleSetFileName = new TypedIOPort(this, "ruleSetFileName", true, false);
    ruleSetFileName.setTypeEquals(BaseType.STRING);
    input.setTypeEquals(BaseType.STRING);
    output.setTypeEquals(BaseType.STRING);

    omissionValue = new TypedIOPort(this, "omissionValue", false, true);
    omissionValue.setTypeEquals(BaseType.DOUBLE);
    commissionValue = new TypedIOPort(this, "commissionValue", false, true);
    commissionValue.setTypeEquals(BaseType.DOUBLE);
    outputRuleSetFileName = new TypedIOPort(this, "outputRuleSetFileName", false, true);
    outputRuleSetFileName.setTypeEquals(BaseType.STRING);

  }

  
  /**
   *
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    String  ruleSetNameStr = "";
    super.fire();
    
    if (pointFileName.getWidth()>0) { // has a connection
      if (pointFileName.hasToken(0)) {  // has a token
        StringToken inputFileToken = (StringToken)pointFileName.get(0);
        String  inputFileNameStr = inputFileToken.stringValue();
        File pointFile = new File(inputFileToken.stringValue());
        FileReader inReader = null;
        BufferedReader bufReader = null;
        try{
         inReader = new FileReader(pointFile);
          bufReader = new BufferedReader(inReader);
          pointList = getPoints(bufReader);
        } catch (Exception ee) {
          System.out.println("Exception reading points!");
        }
      }
    }

    if (ruleSetFileName.getWidth()>0) { // has a connection
      if (ruleSetFileName.hasToken(0)) {  // has a token
        StringToken ruleSetFileToken = (StringToken)ruleSetFileName.get(0);
        ruleSetNameStr = ruleSetFileToken.stringValue();
	// on Windows machines, one sometimes get a file path with a mixture of '/' and '\' symbols
	// convert all '\' to '/'
	ruleSetNameStr = ruleSetNameStr.replace('\\','/');
      }
    }
    
    try
    {
      if (input.getWidth()>0) { //has a connection
        if (input.hasToken(0)) { // has a token
          String ascfilename = ((StringToken)input.get(0)).stringValue();
      
          File file = new File(ascfilename);
      
          if(file.exists()){
            inputGrid = new Grid(file);
            System.out.println("nrows: "+inputGrid.nrows);
            System.out.println("ncols: "+inputGrid.ncols);
            System.out.println("delx: "+inputGrid.delx);
            System.out.println("dely: "+inputGrid.dely);
          
            double fnm = inputGrid.getFractionPixelsWithValue(hit_value, miss_distance);
            java.lang.Double commissionD = new java.lang.Double(fnm);
            String commission = commissionD.toString();
        
            int hitCnt = 0;
            if (pointList!=null) {
              for (int i=0;i<pointList.size();i++) {
                double x2 = ((Point2D)pointList.elementAt(i)).getX();
                double y2 = ((Point2D)pointList.elementAt(i)).getY();
                double val = inputGrid.interpValue(x2, y2, Grid.NEAREST_NEIGHBOR);
                if ((Math.abs(val-hit_value))<miss_distance) {
                  hitCnt++;
                } else {
                   System.out.println("no hit at i = "+i+" --x="+x2+" --y="+y2);
                }
              }
            }
            java.lang.Double omissionD = new java.lang.Double((double)(pointList.size()-hitCnt)/pointList.size());
            String omission = omissionD.toString();
            String outstring = omission+"\t"+commission+"\t"+ruleSetNameStr;
            output.broadcast(new StringToken(outstring));
            
            omissionValue.broadcast(new DoubleToken(omissionD.doubleValue()));
            commissionValue.broadcast(new DoubleToken(commissionD.doubleValue()));
            outputRuleSetFileName.broadcast(new StringToken(ruleSetNameStr));
           }
           else
           {
             throw new IllegalActionException("Input file " + ascfilename +
                " does not exist.");
           }
      
        }
      }
    }
      catch(Exception eee)
    {
      throw new IllegalActionException("Problem Reading File");
    }
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire() throws IllegalActionException 
  {
    inputGrid.delete(); // remove potentially large data storage associated with the inputGrid
    return super.postfire();
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }
  
  private Vector getPoints(BufferedReader br) {
    String cachedLine = "";  
    double xval = 0.0;
    double yval = 0.0;
    Vector pts = new Vector();
    // unsure exactly how many point lines there are
    // but each line should have only two tokens 
    boolean eoh = false;
    while (!eoh) {
      try{
        cachedLine = br.readLine();
//System.out.println("cachedLine: "+cachedLine);
      } catch (Exception w) {
        System.out.println("error reading next line in points file!");
        eoh = true;
      }
      if (cachedLine==null) return pts;
      if (cachedLine.trim().length()>0) {
        StringTokenizer st = new StringTokenizer(cachedLine);
        int cnt = st.countTokens();  // should be only 2
        if (cnt != 2) eoh = true;
        String firstToken = st.nextToken().trim();
        String secondToken = st.nextToken().trim();
        try {
          xval = java.lang.Double.parseDouble(firstToken);
          yval = java.lang.Double.parseDouble(secondToken);
        }
        catch (Exception e) {
          eoh = true;
        }
        if (!eoh) {
          java.awt.geom.Point2D.Double pt2D = new java.awt.geom.Point2D.Double(xval, yval);
          pts.addElement(pt2D);
        }  
      }
    }
    return pts;
  }
  

  private Point2D findFirstPoint(Vector pts) {
    double minx;
    Point2D fp = (Point2D)pts.elementAt(0);
    minx = fp.getX();
    for (int i=1;i<pts.size();i++) {
      double tempx = ((Point2D)pts.elementAt(i)).getX();  
//System.out.println("i: "+i+"    tempx: "+tempx+ "    minx: "+minx);
      if (tempx<minx) {
        fp = (Point2D)pts.elementAt(i); 
        minx = tempx; 
      }
    }
//System.out.println("MinX: "+fp.getX()+"   MinY: "+fp.getY());
    return fp;
  }
  
  
}
