/* Translating a moml description into actor relations axioms.

@Copyright (c) 2002-2003 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the
above copyright notice and the following two paragraphs appear in all
copies of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

                                                PT_COPYRIGHT_VERSION 2
                                                COPYRIGHTENDKEY
*/

package org.geon;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Iterator;

import ptolemy.actor.CompositeActor;
import ptolemy.actor.Director;
import ptolemy.actor.IOPort;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedCompositeActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.data.expr.FileParameter;
import ptolemy.data.expr.Variable;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.Type;
import ptolemy.data.type.TypeLattice;
import ptolemy.graph.CPO;
import ptolemy.graph.Inequality;
import ptolemy.graph.InequalityTerm;
import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Port;
import ptolemy.kernel.Relation;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.Location;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.StringAttribute;
import ptolemy.moml.MoMLParser;


//////////////////////////////////////////////////////////////////////////
////MomlToRelational
/**
 * Parsing the moml file into a container using MoMLParser. Then extracting the
 * schema infomration.
 *
 * director(name, class)
 * actor(name, class)
 * actorPort(actor, portId)
 * actorParam(actor, paramId)
 * portIOType(portId, type) - type = {in, out, inout}
 * portDataType(portId, dataType)
 * paramValue(paramId, type, value)
 *
 * contains(compositeEntity, Entity) - save heirarchical information within the schema.
 *
 * from this a connection between an output and an input port can be inferred.
 *
 * @author Efrat Jaeger
 * @version $Id: MomlToRelational.java,v 1.14 2005/11/01 20:39:08 ruland Exp $
 * @since Ptolemy II 3.0.2
 *
*/
public class MomlToRelational extends TypedAtomicActor {

	 /** Construct an actor with the given container and name.
	 *  @param container The container.
	 *  @param name The name of this actor.
	 *  @exception IllegalActionException If the actor cannot be contained
	 *   by the proposed container.
	 *  @exception NameDuplicationException If the container already has an
	 *   actor with this name.
	 */

	public MomlToRelational(CompositeEntity container, String name)
	        throws NameDuplicationException, IllegalActionException  {

	    super(container, name);

	    relations = new TypedIOPort(this, "relations", false, true);
	    // Set the type constraint.
	    relations.setTypeEquals(BaseType.STRING);

	    momlFile = new FileParameter(this, "momlFile");

	}

	///////////////////////////////////////////////////////////////////
	////                     ports and parameters                  ////

	/** The relations.
	 */
	public TypedIOPort relations;

	/** The moml file.
	 */
	public FileParameter momlFile;

	  	///////////////////////////////////////////////////////////////////
	 	////                         public methods                    ////

        public void fire() throws IllegalActionException {

            try {
            	File _momlFile = momlFile.asFile();
	        	String path = _momlFile.getAbsolutePath();
	        	URL _momlUrl = _momlFile.toURL();

	        	_toplevel = _parse(_momlUrl);
	        	_workspace = _toplevel.getFullName();
	        	_workspaceLen = _workspace.length()+1;

	        	//String desc = _toplevel.description();

	           	CompositeActor CE = (CompositeActor) _toplevel;
	           	_getParams(CE);
	        	String entities = getEntities(CE);

	        	String result =  "%director(name, class)\r\n" + _directors;
				result += "\r\n%actor(name, class)\r\n" + entities;
				result += "\r\n%contains(container, contained)\r\n" + _containment; // + rels;
	        	result += "\r\n%actorPorts(actorName, portName)\r\n" + _actorPorts;
	        	result += "\r\n%portIOType(portName, in/out, single/multi)\r\n" + _portIOType;
	        	result += "\r\n%portDataType(portName, datatype)\r\n" + _portDataType;
	        	result += "\r\n%actorParmeter(actorName, parameterName)\r\n" + _actorParams;
	        	result += "\r\n%parameterValue(parameterName, type, value)\r\n"+ _parameterValues;
	        	result += "\r\n%relation(name, type)\r\n" + _relations;
	        	result += "\r\n%link(portName, relationName)\r\n"+ _links;
	        	result += "\r\n%typeConstraint(port, lowesetConnectType, highestConnectedType)\r\n";
	        	result += _portRange;
	        	result += "\r\n%typeConstraint(port, lesserTermClass, lesserTermName, greaterTermClass, greaterTermName)\r\n";
	        	result += _typeConstraints;
	        	result += "\r\n%typeConstraintValue(port, lesserValue, greaterValue)\r\n";
	        	result += _typeConstraintValues;
	        	result += "\r\n%layout(object, location)\r\n";
	        	result += _layout;
	        	//String connectionRule = "connection(X,Y) :- "
	        	relations.broadcast(new StringToken(result));

	        	_toplevel = null;
            } catch (IOException io) {

            }
        }

    /** Initialize global variables.
     */
    public void initialize() throws IllegalActionException {
        super.initialize();
        _actorParams = "";
        _actorPorts = "";
        _container = "";
        _containment = "";
    	_directors = "";
    	_layout= "";
    	_links = "";
    	_parameterValues = "";
    	_portDataType = "";
    	_portIOType = "";
    	_portRange = "";
    	_relations = "";
    	_typeConstraints = "";
    	_typeConstraintValues = "";
    }

    public boolean postfire() {
    	_actorParams = "";
    	_actorPorts = "";
    	_container = "";
    	_containment = "";
    	_directors = "";
    	_layout= "";
    	_links = "";
    	_parameterValues = "";
    	_portDataType = "";
    	_portIOType = "";
    	_portRange = "";
    	_relations = "";
    	_typeConstraints = "";
    	_typeConstraintValues = "";
    	return false;
    }

    ///////////////////////////////////////////////////////////////////
    ////                        private methods                     ////

	private String getEntities(CompositeActor CE) {
    	Iterator ents = CE.entityList().iterator();
    	String axiom = "";
    	Director director = CE.getDirector();
    	if (director == null) {
    		return "no top level director!";
    	}
    	Location locAtt = (Location) director.getAttribute("_location");
    	String loc = locAtt.getExpression();

    	_container = CE.getFullName();
    	_directors += "director('";
    	_layout += "layout('" ;
    	if (_container.length() > _workspaceLen) {
    		_container = _container.substring(_workspaceLen);
    	   	_directors += _container + ".";
    	   	_layout += _container + ".";
    	}
		_directors += director.getName() + "','";
    	_directors += director.getClass().getName() + "').\r\n";

		_layout += director.getName() + "','location(";
    	_layout += loc + ")').\r\n";

    	_getRelations(CE);

    	while (ents.hasNext()) {
    		Object obj = ents.next();
    		ComponentEntity CA = (ComponentEntity) obj;
    		String container = CA.getContainer().getFullName();
        	if (container.length() > _workspaceLen) {
        		container = container.substring(_workspaceLen);
        	}
    		_containment += "contains('" + container + "','";
    		_containment += CA.getName() + "').\r\n";
    		_actorPorts = _getPorts(CA);
    		_getParams(CA);
    		//System.out.println(obj.getClass().getName());
    		if (CA instanceof TypedCompositeActor) {
    			CompositeActor inner = (CompositeActor) CA;
    			axiom += getEntities(inner);
    			//TODO: add composite entity name??
    		}
    		//System.out.println(CA.getClass());
    		//System.out.println(CA.getName());
    		axiom += "actor('" + CA.getFullName().substring(_workspaceLen) + "','";
    		axiom += obj.getClass().getName() + "').\r\n";

    		locAtt = (Location) CA.getAttribute("_location");
    		loc = locAtt.getExpression();

      		_layout += "layout('" + CA.getFullName().substring(_workspaceLen);
    		_layout += "','location(" + loc + ")').\r\n";
    	}
    	return axiom;

    }

	/** Returns all information about an actor's parameters */
	private void _getParams(ComponentEntity CE) {
		Iterator props = CE.attributeList().iterator();
		while (props.hasNext()) {
			Attribute property =  (Attribute)props.next();
			String propName = property.getName();
                        String attClass = property.getClass().getName();
			if (!propName.startsWith("_") && !attClass.toLowerCase().endsWith("director")) {
				int ind = attClass.lastIndexOf(".");
				String shortClass = attClass.substring(ind + 1);
				String value = "";
				if (shortClass.startsWith("StringAtt") || shortClass.startsWith("FileAtt")) {
					StringAttribute stringAtt = (StringAttribute) property;
					value = stringAtt.getExpression();
				} else if (shortClass.equals("Attribute")) {
					continue;
				} else {
					Variable param = (Variable) property;
					value = param.getExpression();
				}
				//TODO: ADD IF FULLNAME > WORKSPACE LEN!!
				String actorName = CE.getFullName().substring(_workspaceLen);
				_actorParams += "actorParmeter('";
				_actorParams += actorName;
				_actorParams += "','" + propName + "').\r\n";
				_parameterValues += "parameterValue('" + actorName;
				_parameterValues += "." + propName + "','" + attClass;
				_parameterValues += "','" + value + "').\r\n";
			}
		}

	}

	/** Returns all information about an actor's ports */
	private String _getPorts(ComponentEntity CE) {
		Iterator ports = CE.portList().iterator();
		while (ports.hasNext()) {
			TypedIOPort p = (TypedIOPort)ports.next();
			String actorName = CE.getFullName().substring(_workspaceLen);
			String portName = actorName + "." + p.getName();

			String tgStr, tlStr;
			Iterator typeConstraints = p.typeConstraintList().iterator();
			while (typeConstraints.hasNext()) {
				Inequality ie = (Inequality)typeConstraints.next();
				InequalityTerm tg = ie.getGreaterTerm();
				InequalityTerm tl = ie.getLesserTerm();
				NamedObj tgObj = (NamedObj) tg.getAssociatedObject();
				String tgClassName = tgObj.getClass().getName();
				String tgVarName = tgObj.getFullName().substring(_workspaceLen);
				NamedObj tlObj = (NamedObj) tl.getAssociatedObject();
				String tlClassName = tlObj.getClass().getName();
				String tlVarName = tlObj.getFullName().substring(_workspaceLen);

				_typeConstraints += "typeConstraint('" + portName + "','";
				_typeConstraints += tlClassName + "','" + tlVarName + "','";
				_typeConstraints += tgClassName + "','" + tgVarName + ").\r\n";

				try {
					tlStr = tl.getValue().toString();
					tgStr = tg.getValue().toString();
					_typeConstraintValues += "typeConstraintValue('" + portName + "','";
					_typeConstraintValues += tl.getValue() + "','" + tg.getValue() +"').\r\n";
				} catch (IllegalActionException e) {
					e.printStackTrace();
				}

			}
			Type higher = BaseType.UNKNOWN;
			Type lower = BaseType.GENERAL;
			Iterator conPorts = p.connectedPortList().iterator();
			//The type range is between the lowest and highest.
			int compare;
			while (conPorts.hasNext()) {
				TypedIOPort conP = (TypedIOPort)conPorts.next();
				Type conType = conP.getType();
		        compare = TypeLattice.compare(conType,lower);
		        if (compare == CPO.LOWER) {
		        	lower = conType;
		        }
		        compare = TypeLattice.compare(higher, conType);
		        if (compare == CPO.LOWER) { 
		        	higher = conType;
		        }
			}
			compare = TypeLattice.compare(lower, higher);
			if (compare == CPO.HIGHER) {
				higher = lower;
			}
	/*		try { 
				if (!tgStr.equals("general") && !tgStr.equals("unknown")) {
					higher = (Type) tg.getValue();
				}
				if (tl.getValue() != BaseType.GENERAL && tl.getValue() != BaseType.UNKNOWN) {
					lower = (Type) tl.getValue();
				}
			} catch (Exception ex) {
				ex.printStackTrace();
			}*/
			_portRange += "portTypeRange('" + portName + "','";
			_portRange += lower + "','" + higher + "').\r\n";


			_actorPorts += "actorPorts('" + actorName + "','";
			_actorPorts += portName + "').\r\n";
			String ioType = "";
			if (p.isInput() && p.isOutput()) {
				ioType = "inOut";
			} else if (p.isInput()) {
				ioType = "in";
			} else if (p.isOutput()) {
				ioType = "out";
			} else System.out.println("ERROR: no port IO type!");
			String pchannels = "singlePort";
			if (p.isMultiport()) {
				pchannels = "multiPort";
			}
			_portIOType += "portIOType('" + portName;
			_portIOType += "','" + ioType;
			_portIOType += "','" + pchannels + "').\r\n";
			_portDataType += "portDataType('" + portName;
			_portDataType += "','" + p.getType() + "').\r\n";

		}
		return _actorPorts;
	}
    private void _getRelations(CompositeEntity CE) {

    	String container = CE.getFullName();
    	if (container.length() > _workspaceLen) {
    		container = container.substring(_workspaceLen);
    	}
        Iterator rels = CE.relationList().iterator();
    	while (rels.hasNext()) {
    		Relation rel = (Relation) rels.next();
    		String relName = rel.getFullName().substring(_workspaceLen);
    		String relClass = rel.getClass().getName();

    		_relations += "relation('" + relName + "','";
    		_relations += relClass + "').\r\n";
    		String vertex = "";
    		Iterator relAtts = rel.attributeList().iterator();
/*    		while (relAtts.hasNext()) {
    			Vertex relAtt = (Vertex) relAtts.next();
    			//Object obj = (Vertex) relAtts.next();
    			//if (obj != null) {
    					//Vertex relAtt = (Vertex) obj;   
                       // vertex = "vertex(" + relAtt.getExpression() + ")";
                        //_layout += "layout('" + relName + "','" + vertex + "').\r\n";
                //}
    		}
*/
    		Iterator ports = rel.linkedPortList().iterator();
    		while (ports.hasNext()) {
    			Port p = (IOPort)ports.next();
    			String portName = p.getFullName().substring(_workspaceLen);
    			_links += "link('" + portName + "','";
    			_links += relName + "').\r\n";
    		}
    	}

    }

    private NamedObj _parse(URL _momlUrl) {
    	MoMLParser parser = new MoMLParser();
    	parser.reset();
    	try {
    		_toplevel = parser.parse(_momlUrl, _momlUrl);
    	} catch (Exception ex) {
    		System.out.println("moml parser exception" + ex.getMessage());
    	}
    	return _toplevel;
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private members                   ////

    /** actorPorts axioms */
    private String _actorPorts = "";

    /** actorParameters axioms */
    private String _actorParams = "";

    /** current entity's container */
    private String _container = "";

    /** the containment relation */
    private String _containment = "";

    /** the directors */
    private String _directors = "";

	/** saving the layout of the workflow */
	private String _layout= "";

    /** link axioms */
    private String _links = "";

    /** the parameters classes and values */
    private String _parameterValues = "";

    /** portDataType axioms */
    private String _portDataType = "";

    /** portIOType axioms */
    private String _portIOType = "";

    /** portIOType axioms */
    private String _portRange = "";

    /** relation axioms */
    private String _relations = "";

    /** top level entity */
    private NamedObj _toplevel = null;

    /** type constraints axioms */
    private String _typeConstraints = "";

    /** type constraint values axioms */
    private String _typeConstraintValues = "";

    /** workspace name */
    private String _workspace;

    /** length of workspace name */
    private int _workspaceLen;

}