/**
 *  '$RCSfile: ResultTreeRebuilder.java,v $' Copyright: 2004 Regents of the
 *  University of California and the National Center for Ecological Analysis and
 *  Synthesis 
 * '$Author: berkley $' 
 * '$Date: 2006/03/13 20:00:54 $' 
 * '$Revision: 1.5 $' 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version. This program is distributed in the hope that it
 *  will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 *  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *  Public License for more details. You should have received a copy of the GNU
 *  General Public License along with this program; if not, write to the Free
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 *  USA
 */
package org.kepler.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.io.IOException;
import java.util.Iterator;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.tree.TreePath;

import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Entity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.Workspace;
import ptolemy.moml.EntityLibrary;
import ptolemy.vergil.tree.PTree;
import ptolemy.vergil.tree.VisibleTreeModel;

/**
 *  This class builds the search results by traversing the tree and trimming any
 *  sub-nodes that do not have a result in them. This leaves a minimum tree with
 *  only the search results present.
 *
 *@author     Chad Berkley
 *@author     Shawn Bowers (... last editor ...)
 *@created    February 17, 2005
 */
public class ResultTreeRebuilder extends LibrarySearchResultPane
{
  private VisibleTreeModel trimmedLibrary;
  private EntityLibrary newRoot;
  //private AnnotatedPTree resultsTree;
  private PTree resultsTree;
  private Workspace workspace;
  private JPanel resultCounterPane;
  private JLabel resultCounterLabel;


  /**
   *  the constructor passes in the library to highlight the results in and the
   *  results to highlight. if results is null, the tree is built fully
   *  collapsed with no highlights.
   *
   *@param  library                     the library to highlight the results in
   *@param  results                     the results to highlight
   *@exception  IllegalActionException  Description of the Exception
   */
  public ResultTreeRebuilder(PTree library, LibrarySearchResults results)
    throws IllegalActionException
  {
    super(library, results);
    this.workspace = ((CompositeEntity) library.getModel().getRoot()).workspace();
    /*
     *  resultCounterPane = new JPanel();
     *  resultCounterPane.setPreferredSize(new Dimension(20, 200));
     *  resultCounterPane.setLayout(new BorderLayout());
     */
  }


  /**
   *  this method allows the search results to be updated in the panel
   *
   *@param  results                     the results to update to
   *@exception  IllegalActionException  Description of the Exception
   */
  public void update(LibrarySearchResults results)
    throws IllegalActionException
  {
    this.results = results;
    int resultcount;
    if (results == null || results.size() == 0)
    {
      resultcount = 0;
    }
    else
    {
      resultcount = results.size();
    }

    // clearTree();
    this.removeAll();

    //add the results if there are any
    if (resultcount > 0)
    {
      //add the results tree.
      this.newRoot = new EntityLibrary(workspace);

      try
      {
        newRoot.setName("Search Results");
      }
      catch (IllegalActionException iae)
      {
        throw iae;
      }
      catch (NameDuplicationException nde)
      {
        throw new IllegalActionException("name duplication exception: " + nde);
      }
      buildResultTree();
      trimmedLibrary = new VisibleTreeModel(newRoot);
      resultsTree = new AnnotatedPTree(trimmedLibrary, this);
      //resultsTree = new PTree(trimmedLibrary);
      JScrollPane newpane = new JScrollPane(resultsTree);
      newpane.setPreferredSize(new Dimension(200, 200));
      this.add(newpane, BorderLayout.CENTER);
      expandAll(resultsTree);
    }
    else
    {
      //if there are no results, just add the library back
      this.add(new JScrollPane(library), BorderLayout.CENTER);
    }

    //add the search results counter stuff
    resultCounterPane = new JPanel();
    resultCounterPane.setLayout(new BorderLayout());
    resultCounterLabel = new JLabel(resultcount + " results found.");
    resultCounterPane.removeAll();
    resultCounterPane.add(resultCounterLabel, BorderLayout.CENTER);
    this.add(resultCounterPane, BorderLayout.SOUTH);

    this.repaint();
    this.validate();
  }



  /**
   *  clears the tree of any items contained by newRoot
   *
   *@exception  IllegalActionException  Description of the Exception
   */
  private void clearTree()
    throws IllegalActionException
  {
    if (newRoot != null && newRoot.entityList() != null)
    {
      for (Iterator i = newRoot.containedObjectsIterator(); i.hasNext(); )
      {
        CompositeEntity ce = (CompositeEntity) i.next();
        try
        {
          ce.setContainer(null);
          executeChangeRequest(ce, newRoot);
        }
        catch (IllegalActionException iae)
        {
          throw new IllegalActionException("Cannot remove entity from search " +
              "result tree");
        }
        catch (NameDuplicationException nde)
        {
          throw new IllegalActionException("cannot set container because the name " +
              "already exists: " + nde);
        }
        catch (IOException ioe)
        {
          throw new IllegalActionException("IO Exception while removing entity " +
              " search result tree: " + ioe);
        }
        catch (Exception e)
        {
          //do nothing...just go onto the next node
        }
      }
    }
  }


  /*
   *  build the result tree
   */
  /**
   *  Description of the Method
   *
   *@exception  IllegalActionException  Description of the Exception
   */
  private void buildResultTree()
    throws IllegalActionException
  {
    for (int i = 0; i < results.size(); i++)
    {
      //iterate over the treepaths in results
      TreePath path = results.getTreePath(i);
      // ComponentEntity ceClone = null;
      NamedObj ceClone = null;

      //get the leaf node of the result path
      // ComponentEntity ce = (ComponentEntity)path.getPathComponent(path.getPathCount() - 1);
      NamedObj ce = (NamedObj) path.getPathComponent(path.getPathCount() - 1);

      if (ce instanceof EntityLibrary)
      {
        //folder (EntityLibrary)
        if (ceClone == null)
        {
          //since the result is an entitylibrary, we want to copy the
          //folder and it's contents because the folder doesn't do much good
          //without its contents
          ceClone = cloneEntity(ce);
        }
      }
      else
      {
        //leaf (AtomicActor)
        ceClone = cloneEntity(ce);
      }

      NamedObj currentEntity = ceClone;
      for (int j = path.getPathCount() - 2; j >= 0; j--)
      {
        NamedObj nobj = getDeepEntity((NamedObj) path.getPathComponent(j), newRoot);
        
        CompositeEntity pathEntity = (CompositeEntity) path.getPathComponent(j);
        CompositeEntity newEntity;
        
        if (nobj == null && !pathEntity.getName().equals("actor library"))
        {
          newEntity = copyCompositeEntity(pathEntity);
        }
        else
        {
          newEntity = (CompositeEntity) nobj;
        }
        
        try
        {
          if (j == 0)
          {
            //the root node
            if (getDeepEntity(currentEntity, newRoot) == null)
            {
              setContainer(newRoot, currentEntity);
              executeChangeRequest(currentEntity, newRoot);
            }
          }
          else
          {
            //all the others
            if (getDeepEntity(currentEntity, newEntity) == null)
            {
              setContainer(newEntity, currentEntity);
              executeChangeRequest(currentEntity, newEntity);
            }
          }
        }
        catch (IllegalActionException iae)
        {
          throw new IllegalActionException("cannot build search result tree: " + iae);
        }
        catch (NameDuplicationException nde)
        {
          //FIXME: ignore this for now.  I'm not sure why this works
          //throw new IllegalActionException("cannot add entity to the search " +
          //  "result container because the name already exists: " + nde);
        }
        catch (IOException ioe)
        {
          throw new IllegalActionException("IO Exception: " + ioe);
        }
        catch (Exception e)
        {
          //FIXME: These exceptions are being thrown from the changeRequest.
          //I'm not sure why they are being thrown or why I can't catch
          //them as an IllegalActionException so......
          //do nothing.  just go onto the next path..
          System.out.println("EXCEPTION CAUGHT: " + e.getMessage());
        }

        currentEntity = newEntity;
      }
    }
  }


  /**
   *  puts entity in container
   *
   *@param  container                     The new container value
   *@param  entity                        The new container value
   *@exception  NameDuplicationException  Description of the Exception
   *@exception  IllegalActionException    Description of the Exception
   */
  private static void setContainer(CompositeEntity container, NamedObj entity)
    throws NameDuplicationException, IllegalActionException
  {
    if (entity instanceof Attribute)
    {
      ((Attribute) entity).setContainer(container);
    }
    else if (entity instanceof ComponentEntity)
    {
      ((ComponentEntity) entity).setContainer(container);
    }
  }


  /**
   *  search entityToSearch for entity. this searches all entities contained by
   *  entityToSearch (deep search). returns null if entity does not exist in
   *  entityToSearch
   *
   *@param  entity          Description of the Parameter
   *@param  entityToSearch  Description of the Parameter
   *@return                 The deepEntity value
   */
  private static NamedObj getDeepEntity(NamedObj entity, CompositeEntity entityToSearch)
  {
    for (Iterator i = entityToSearch.entityList().iterator(); i.hasNext(); )
    {
      //search each enity in the entityToSearch
      Entity e = (Entity) i.next();
      if (e.getName().equals(entity.getName()))
      {
        return e;
      }
      else
      {
        if (e instanceof CompositeEntity)
        {
          //if we find another compositeentity, go into it and search
          NamedObj returnEntity = getDeepEntity(entity, (CompositeEntity) e);
          if (returnEntity != null)
          {
            //if we find something, return it back through the recursion
            return returnEntity;
          }
        }
      }
    }

    return null;
    //no results found
  }


  /**
   *  copy the entity in name only unless the deepCopy flag is true, then copy
   *  the entity in name and copy it's contents as well
   *
   *@param  entity                      Description of the Parameter
   *@return                             Description of the Return Value
   *@exception  IllegalActionException  Description of the Exception
   */
  private EntityLibrary copyCompositeEntity(CompositeEntity entity)
    throws IllegalActionException
  {
    try
    {
      return new EntityLibrary(new CompositeEntity(this.workspace), entity.getName());
    }
    catch (IllegalActionException iae)
    {
      throw new IllegalActionException("cannot copy composite entity: " + iae);
    }
    catch (NameDuplicationException nde)
    {
      throw new IllegalActionException("cannot set container because the name " +
          "already exists: " + nde);
    }
  }


  /**
   *  request a change to the model
   *
   *@param  changedEntity                 Description of the Parameter
   *@param  targetEntity                  Description of the Parameter
   *@exception  NameDuplicationException  Description of the Exception
   *@exception  IllegalActionException    Description of the Exception
   *@exception  IOException               Description of the Exception
   */
  //private static void executeChangeRequest(ComponentEntity changedEntity, ComponentEntity targetEntity)
  private static void executeChangeRequest(NamedObj changedEntity, NamedObj targetEntity)
    throws NameDuplicationException, IllegalActionException, IOException
  {
    // StringWriter buffer = new StringWriter();
    // changedEntity.exportMoML(buffer, 1);
    // ChangeRequest request = new MoMLChangeRequest(changedEntity, targetEntity,
    // buffer.toString());
    // targetEntity.requestChange(request);
  }


  /**
   *  clone the entity into the new workspace.
   *
   *@param  entity                      Description of the Parameter
   *@return                             Description of the Return Value
   *@exception  IllegalActionException  Description of the Exception
   */
  //    private ComponentEntity cloneEntity(ComponentEntity entity) throws IllegalActionException {
  private NamedObj cloneEntity(NamedObj entity)
    throws IllegalActionException
  {
    try
    {
      return (NamedObj) entity.clone(this.workspace);
    }
    catch (java.lang.CloneNotSupportedException cnse)
    {
      throw new IllegalActionException("clone not supported: " + cnse);
    }
  }


  /**
   *  A factory that creates the searcher to search the library
   *
   *@author     berkley
   *@created    February 17, 2005
   */
  public static class Factory extends LibrarySearchResultPaneFactory
  {
    /**
     *  Create an factory with the given name and container.
     *
     *@param  container                     The container.
     *@param  name                          The name of the entity.
     *@exception  IllegalActionException    If the container is incompatible
     *      with this attribute.
     *@exception  NameDuplicationException  If the name coincides with an
     *      attribute already in the container.
     */
    public Factory(NamedObj container, String name)
      throws IllegalActionException, NameDuplicationException
    {
      super(container, name);
    }


    /**
     *  creates a ResultTreeBuilder and returns it.
     *
     *@param  library                     Description of the Parameter
     *@param  results                     Description of the Parameter
     *@return                             A new LibraryPane that displays the
     *      library
     *@exception  IllegalActionException  Description of the Exception
     */
    public LibrarySearchResultPane createLibrarySearchResultPane(PTree library, LibrarySearchResults results)
      throws IllegalActionException
    {
      return new ResultTreeRebuilder(library, results);
    }
  }
}

