/**
 *  '$RCSfile: KSWBuilder.java,v $'
 *  '$Author: berkley $'
 *  '$Date: 2006/03/28 00:47:17 $'
 *  '$Revision: 1.25 $'
 *
 *  For Details:
 *  http://www.kepler-project.org
 *
 *  Copyright (c) 2004 The Regents of the
 *  University of California. All rights reserved. Permission is hereby granted,
 *  without written agreement and without license or royalty fees, to use, copy,
 *  modify, and distribute this software and its documentation for any purpose,
 *  provided that the above copyright notice and the following two paragraphs
 *  appear in all copies of this software. IN NO EVENT SHALL THE UNIVERSITY OF
 *  CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
 *  OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 *  DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY
 *  DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 *  SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.ksw;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.jar.JarOutputStream;

import org.kepler.moml.NamedObjId;
import org.kepler.objectmanager.ActorMetadata;
import org.kepler.objectmanager.cache.ActorCacheObject;
import org.kepler.objectmanager.cache.CacheManager;
import org.kepler.objectmanager.lsid.KeplerLSID;
import org.kepler.objectmanager.lsid.LSIDTree;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedCompositeActor;
import ptolemy.kernel.Entity;
import ptolemy.kernel.util.IllegalActionException;

/**
 *  Class to create KSW files from within kepler
 */
public class KSWBuilder
{
  File workDir;
  ActorMetadata am;
  boolean registerLSID;
  KeplerLSID karLSID = null;
  
  /**
   * Constructor.  takes in a CompositeEntity and a boolean that tells whether the
   * ksw file should be created by reference or by archive.  each object in
   * the CompositeEntity will be added to the ksw file.
   * @param entity the entity to build the ksw from.
   * @param archive true if you want to build an archive ksw, false if you
   *  want to build a reference ksw
   * @param filepath the path at which to build the ksw file
   */
  public KSWBuilder(Entity entity, boolean archive, File kswFile)
    throws IllegalActionException
  {
    this(entity, archive, true, kswFile, null);
  }
  
  /**
   * This constructor allows you to specify the lsid for the kar file.
   */
  public KSWBuilder(Entity entity, boolean archive, KeplerLSID karLSID, 
    File kswFile) throws IllegalActionException
  {
    this(entity, archive, false, kswFile, karLSID);
  }
  
  /**
   * the same contructor as below, but it make the kswBuilder create the lsid
   * for the kar file.
   */
  public KSWBuilder(Entity entity, boolean archive, boolean registerLSID, 
    File kswFile) throws IllegalActionException
  {
    this(entity, archive, registerLSID, kswFile, null);
  }

  /**
   * Constructor.  takes in a CompositeEntity and a boolean that tells whether the
   * ksw file should be created by reference or by archive.  each object in
   * the CompositeEntity will be added to the ksw file.
   * @param entity the entity to build the ksw from.
   * @param archive true if you want to build an archive ksw, false if you
   *  want to build a reference ksw
   * @param registerLSID true if you want new lsids registered with the cache.
   * If you are going to eventually cache the created kar file, set registerLSID 
   * to false.
   * @param filepath the path at which to build the ksw file
   */
  public KSWBuilder(Entity entity, boolean archive, boolean registerLSID, 
    File kswFile, KeplerLSID lsid)
    throws IllegalActionException
  {
    this.karLSID = lsid;
    try
    {
      this.registerLSID = registerLSID;
      Hashtable kswItems = new Hashtable();
      KSWManifest manifest = new KSWManifest();
      if(archive)
      {
        System.out.println("Creating archive KAR file at " + 
          kswFile.getAbsolutePath());
        createArchiveKSW(entity, kswItems, manifest);
      }
      else
      {
        System.out.println("Creating transport KAR file at " +
          kswFile.getAbsolutePath());
        createTransportKSW(entity, kswItems, manifest);
      }
      writeKSWFile(kswItems, manifest, kswFile);
    }
    catch(Exception e)
    {
      throw new IllegalActionException("Error building the KAR file: " +
        e.getMessage());
    }
  }
  
  /**
   * Return the ActorMetadata object used in the kar file.
   * This will return null unless you first call one of the create methods.
   */
  public ActorMetadata getActorMetadata()
  {
    return am;
  }

  /**
   * create a ksw file from an CompositeEntity.
   * The created KSW file will contain the following:
   * 1) references to any java classes (referenced in entities)
   * 2) reference to any data object
   * 3) the workflow (moml model) itself
   * @param entity the CompositeEntity to build the KSW from
   * @param filepath the path where you want to write the ksw file
   * @throws IllegalActionException if anything bad happens
   */
  private void createTransportKSW(Entity entity, Hashtable items, 
    KSWManifest manifest)
    throws IllegalActionException
  {
    try
    {
      if(entity instanceof TypedAtomicActor)
      { //create a ksw file with just this actor in it
        createKSW(entity, items, manifest);
      }
      else if(entity instanceof TypedCompositeActor)
      {
        if(!entity.isClassDefinition())
        { //make sure we have a class here.
          throw new IllegalActionException("You may only create a KAR from " +
            "a TypedComposite Class.  To create a class from your composite, " +
            "right click on the actor and choose \"Convert to Class\" from " +
            "the menu.");
        }
        
        createKSW(entity, items, manifest);
      }
      else
      {
        throw new IllegalActionException("Sorry, KAR creation isn't " +
          "implemented for this type of actor or workflow yet.");
      }
    }
    catch(Exception e)
    {
      e.printStackTrace();
      throw new IllegalActionException("Cannot create transport KAR: " + 
        e.getMessage());
    }
  }
  
  /**
   * create a ksw
   */
  private void createKSW(Entity entity, Hashtable items, KSWManifest manifest)
    throws com.ibm.lsid.LSIDException
  {
    NamedObjId id = (NamedObjId)entity.getAttribute("entityId");
    if(karLSID == null)
    {
      karLSID = LSIDTree.getInstance().getNextObject(
        new KeplerLSID("urn:lsid:kepler-project.org:kar:1:1"), false);
    }
    
    if(registerLSID)
    {
      try
      {
        LSIDTree.getInstance().addLSID(karLSID);
      }
      catch(Exception e)
      {
        System.out.println("Error adding lsid: " + e.getMessage());
        e.printStackTrace();
      }
    };
    manifest.addMainAttribute("lsid", karLSID.toString());
    //get the next revision of the lsid
    KeplerLSID oldLsid = new KeplerLSID(id.getExpression());
    KeplerLSID lsid = LSIDTree.getInstance().findNextRevision(oldLsid);
    am = new ActorMetadata(entity);
    //am.setName(oldAm.getName() + "-" + lsid.getRevision());
    am.setName(entity.getName());
    am.setId(lsid.toString());
    //cache the new AM object
    String actorFilename = am.getId().replaceAll(":", ".") + ".xml";
    KSWEntry entry = new KSWEntry(actorFilename);
    manifest.addEntryAttribute(actorFilename, "lsid", lsid.toString());
    manifest.addEntryAttribute(actorFilename, "type", "actorMetadata");
    items.put(entry, new ByteArrayInputStream(am.toString().getBytes()));
  }

  /**
   * create a ksw file from an CompositeEntity.  This will put all entities
   * within this composite entity that have a class implementation into the
   * ksw file.  The created KSW file will contain the following:
   * 1) any java classes (referenced in entities)
   * 2) any data object
   * 3) the workflow (moml model) itself
   * @param entity the CompositeEntity to build the KSW from
   * @param filepath the path where you want to write the ksw file
   * @throws IllegalActionException if anything bad happens
   */
  private void createArchiveKSW(Entity entity, Hashtable kswItems,
    KSWManifest manifest)
    throws IllegalActionException
  {
    throw new IllegalActionException("Sorry, this isn't implemented yet.  " +
      "Please try creating a transport KAR instead.");
    /*try
    {
      //put the workflow file itself into the vector
      StringReader sr = new StringReader(entity.exportMoML());
      //kswItems.add(sr);
      //this is a hack.  i should be able to tell the type programatically,
      //but i can't figure out how in the moml api
      String type = "workflow";
      if(kswItems.size() == 0)
        type = "class";

      if(entity.getName() == null || entity.getName().equals(""))
      {
        manifest.addEntryAttribute("unnamedWorkflow.xml", "lsid", "");
        manifest.addEntryAttribute("unnamedWorkflow.xml", "type", type);
        KSWEntry entry = new KSWEntry("unnamedWorkflow.xml");
        kswItems.put(entry, sr);
      }
      else
      {
        manifest.addEntryAttribute(entity.getName(), "lsid", "");
        manifest.addEntryAttribute(entity.getName(), "type", type);
        KSWEntry entry = new KSWEntry(entity.getName() + ".xml");
        kswItems.put(entry, sr);
      }

      //write out the java classes, look for data actors, also look for
      //CompositeEntities so we can get their components too.
      Iterator iterator = entity.entityList().iterator();
      while(iterator.hasNext())
      {
        Entity e = (Entity)iterator.next();
        String className = e.getClassName();
        String classFilename = className.replace('.', '/') + ".xml";
        if(className.equals("ptolemy.actor.CompositeActor"))
        { //recurse into the Composite
          createArchiveKSW((CompositeEntity)e, kswItems, manifest);
        }
        //if this is an atomic, find the object in the cache and add it to the
        //jar stream
        String actorId = ((NamedObjId)e.getAttribute("entityId")).getExpression();
        String classId = ((NamedObjId)e.getAttribute("classId")).getExpression();

        //get the actor metadata
        InputStream actorIs = cache.getObjectAsStream(new KeplerLSID(actorId));
        //get the actor class
        InputStream classIs = cache.getObjectAsStream(new KeplerLSID(classId));
        //add the input streams to the items vector
        KSWEntry classEntry = new KSWEntry(classFilename);
        manifest.addEntryAttribute(classFilename, "lsid", classId);
        manifest.addEntryAttribute(classFilename, "type", "class");

        String actorFilename = actorId.replace(':', '.') + ".xml";
        KSWEntry actorEntry = new KSWEntry(actorFilename);
        manifest.addEntryAttribute(actorFilename, "lsid", actorId);
        manifest.addEntryAttribute(actorFilename, "type", "actorMetadata");

        kswItems.put(actorEntry, actorIs);
        kswItems.put(classEntry, classIs);
      }
    }
    catch(Exception e)
    {
      throw new IllegalActionException("There was an error building the " +
        "archive KAR file: " + e.getMessage());
    }*/
  }
  
  /**
   *
   */
  private void writeKSWFile(Hashtable items, KSWManifest manifest, File kswFile)
    throws IOException
  {
    JarOutputStream jos = new JarOutputStream(new FileOutputStream(kswFile),
      manifest);
    Enumeration keys = items.keys();
    while(keys.hasMoreElements())
    {
      KSWEntry entry = (KSWEntry)keys.nextElement();
      jos.putNextEntry(entry);

      if(items.get(entry) instanceof InputStream)
      { //inputstream from a bin file
        byte[] b = new byte[1024];
        InputStream is = (InputStream)items.get(entry);
        int numread = is.read(b, 0, 1024);
        while(numread != -1)
        {
          jos.write(b, 0, numread);
          numread = is.read(b, 0, 1024);
        }
        is.close();
        //jos.flush();
        jos.closeEntry();
      }
      else
      { //stringreader from an xml file
        char[] c = new char[1024];
        StringReader sr = (StringReader)items.get(entry);
        int numread = sr.read(c, 0, 1024);
        while(numread != -1)
        {
          jos.write(new String(c).getBytes(), 0, numread);
          numread = sr.read(c, 0, 1024);
        }
        sr.close();
        jos.flush();
        jos.closeEntry();
      }
    }
    jos.flush();
    jos.close();

    System.out.println("done writing KAR file to " + kswFile.getAbsolutePath());
  }

}

