package org.kepler.objectmanager.cache;

import java.io.InvalidObjectException;
import java.io.ObjectStreamException;
import java.io.Serializable;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;

/**
 * CacheExpiration is an object which defines the expiration policy or time for
 * a CacheObject.  The class allows three different kinds of policies:
 * 
 * Session - Defined by the single instance CacheExpiration.SESSION an object with
 * this expiration policy will expire when the kepler application is terminated.
 * 
 * Never - Defined by the single instance CacheExpiration.NEVER an object with
 * this expiration policy will never expire.
 * 
 * Date - A CacheExpiration can be defined by a java.util.Date object.  A CacheObject
 * with this expiration policy will expire when the current Date exceeds the CacheExpiration date.
 * 
 * It is up to the CacheManager to enforce these policies.  The CacheManager may remove expired
 * objects at startup (in the case of Session expiration), or during the getObject(), putObject() calls.
 * 
 * 
 * @author Kevin Ruland
 *
 */
public class CacheExpiration implements Serializable {

    /*
     * value is the Date representation of the CacheExpiration.  It is marked as
     * transient so standard Java serialization does not emit it in the object stream.
     * The serialized representation is simply a String representation.
     */
    private transient final Date value;

    private final String name;

    private final static String SESSION_STRING = "Session";
    private final static String NEVER_STRING = "Never";
    
    /**
     * Construct a CacheExpiration object only from a String.  This constructor is
     * only to be used to construct SESSION and NEVER instances.
     * @param n
     */
    private CacheExpiration(String n) {
        name = n;
        value = null;
    }

    private CacheExpiration(Date date) throws CacheExpirationException {
        if ( new Date().compareTo(date) > 0 ) {
            throw new CacheExpirationException("Invalid CacheExpiration: date in the past");
        }
        name = DateFormat.getDateInstance().format(date);
        value = date;
    }

    public String toString() {
        return name;
    }

    /**
     * Return true if this has expired by date.
     * @return
     */
    public boolean isExpired() {
        if ( this == SESSION || this == NEVER) {
            return false;
        }
        return ( new Date().compareTo(value) >0 );
    }
    /**
     * Factory mechanism to construct a CacheExpiration object.  If the string is either
     * of the magic strings "Session" or "Never" then return the corresponding single instance
     * object.  Otherwise try to interpret the string as a Date object.  If it is not a valid
     * Date object, raise a CacheExpirationException.
     * 
     * @param s
     * @return
     * @throws CacheExpirationException
     */
    public static CacheExpiration newInstance(String s) throws CacheExpirationException {

        if (s == null || "".equals(s))
        {
            return NEVER;
        }

        if (SESSION_STRING.equalsIgnoreCase(s))
        {
            return SESSION;
        }

        if (NEVER_STRING.equalsIgnoreCase(s))
        {
            return NEVER;
        }

        try
        {
            Date d = DateFormat.getDateInstance().parse(s);
            return new CacheExpiration(d);
        } catch (ParseException e)
        {
            throw new CacheExpirationException("Invalid CacheExpiration: " + s );
        }

    }

    /**
     * Factory instance which always resturns the NEVER Expiration policy.
     * @return
     */
    public static CacheExpiration newInstance() {
        return NEVER;
    }

    public static final CacheExpiration SESSION = new CacheExpiration("Session");

    public static final CacheExpiration NEVER = new CacheExpiration("Never");
    
    /**
     * A custom readResolve method which interprets uses the String representation (in name)
     * and calls the newInstance(String) factory mechansim to return either the static instances
     * or a Date based CacheExpiration.
     * @return
     * @throws ObjectStreamException
     */
    private Object readResolve() throws ObjectStreamException {
        try {
         return newInstance( name );
        }
        catch( CacheExpirationException e) {
            throw new InvalidObjectException(CacheExpiration.class.getName());
        }
    }
}

