/**
 *    '$RCSfile: CacheObject.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/02/03 15:51:10 $'
 *   '$Revision: 1.13 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.objectmanager.cache;

import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;
import java.util.Hashtable;

import org.kepler.objectmanager.lsid.KeplerLSID;

import com.ibm.lsid.LSIDException;

/**
 * Class that represents an object in the ObjectCache.  This class should
 * be extended by each type of object that wants to control its own lifecycle
 * events and serialization events.
 */
public abstract class CacheObject implements CacheObjectInterface, Serializable
{
	protected transient String name;
	protected transient KeplerLSID lsid;
	protected Hashtable attributes = new Hashtable();
    protected transient CacheExpiration expiration;
	
    
	/**
	 * default constructor.  if you use this constrctor, you'll need to set the
	 * name and lsid through the setLSID and setName methods.
	 */
	protected CacheObject()
	{
		//default constructor;
		this.lsid = null;
		this.name = null;
        this.expiration = CacheExpiration.newInstance();
	}
	
	/**
	 * construct a new CacheObject
	 */
	public CacheObject(String name, KeplerLSID lsid)
	{
		this.name = name;
		this.lsid = lsid;
        this.expiration = CacheExpiration.newInstance();
	}
	
	/**
	 * get the name of this object
	 */
	public String getName()
	{
		return name;
	}
	
	/**
	 * get the lsid for this object
	 */
	public KeplerLSID getLSID()
	{
		return lsid;
	}
	
    /**
     * get the expiration
     * @return
     */
    public CacheExpiration getExpiration()
    {
        return expiration;
    }
    
	/**
	 * set the lsid
	 */
	public void setLSID(KeplerLSID lsid)
	{
		this.lsid = lsid;
	}
	
	/**
	 * set the name
	 */
	public void setName(String name)
	{
		this.name = name;
	}
	
    /**
     * set the expiration
     * @return
     */
    public void setExpiration( CacheExpiration e)
    {
        expiration = e;
    }

    /**
	 * set a user configured attribute on the CacheObject
	 */
	public void addAttribute(String name, String value)
	{
		attributes.put(name, value);
	}
	
	/**
	 * get the attribute with the specified name
	 */
	public String getAttribute(String name)
	{
		return (String)attributes.get(name);
	}
	
	/**
	 * remove the attribute with the given name and return it.
	 */
	public void removeAttribute(String name)
	{
		attributes.remove(name);
	}
	
	/**
	 * return the java object associated with this CacheObject
	 */
	public abstract Object getObject();
	
	/**
	 * call back for when this object is added to the cache.  This method should be
	 * overridden by derviced classes if needed. Default action is
	 * to do nothing.
	 */
	public void objectAdded()
	{}
	
	/**
	 * call back for when this object is removed by the user.  This method should be
	 * overridden by derived classes if needed.  Default action is to do nothing.
	 */
	public void objectRemoved()
	{}
	
	/**
	 * call back for when this object is purged by ObjectCache.    This method should be
	 * overridden by derived classes if needed.  Default action is to do nothing.
	 */
	public void objectPurged()
	{}
	
}
