/*
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.objectmanager.lsid;

import java.io.FileInputStream;
import java.io.InputStream;

import org.kepler.objectmanager.ActorMetadata;
import org.kepler.objectmanager.IDNotFoundException;
import org.kepler.objectmanager.IDResolverInterface;
import org.kepler.objectmanager.InvalidMetadataException;
import org.kepler.objectmanager.cache.DataCacheFileObject;
import org.kepler.objectmanager.cache.DataCacheManager;

import com.ibm.lsid.LSIDException;

/**
 * This class resolves LSIDs to various forms.
 *
 *@author     Chad Berkley
 *@created    April 07, 2005
 */
public class LSIDResolver implements IDResolverInterface
{
  /**singleton instance**/
  private static LSIDResolver resolver;
  
  protected LSIDResolver()
  {}

  /**
   * returns the singleton static object of this class
   */
  public static LSIDResolver instanceOf()
  {
    if(resolver == null)
    {
      resolver = new LSIDResolver();
    }
    return resolver;
  }

  /**
   * resolve the id
   * @param id
   */
  public InputStream resolveID(String id) throws IDNotFoundException
  {
    try
    {
      return resolveID(new KeplerLSID(id));
    }
    catch(LSIDException lside)
    {
      throw new IDNotFoundException(lside.getMessage());
    }
  }

  /**
   * resolve a single lsid and return the data stream, if this is a localhost
   * lsid, this will attempt to resolve it from the local cache.
   *
   * @param id the id to resolve
   * @return InputStream the data stream
   * @throws IDNotFoundException
   */
  public InputStream resolveID(KeplerLSID id) throws IDNotFoundException
  {
    try
    {
      if(!id.isLocalhost())
      { //let the lsid resolver do it's job if this is a remote lsid
        com.ibm.lsid.client.LSIDResolver lsidresolver =
          new com.ibm.lsid.client.LSIDResolver(id.getProperLsid());
        InputStream datastream = lsidresolver.getData();
        return datastream;
      }
      else
      {
          throw new IDNotFoundException("Could not resolve LSID " + id.getLsid());
             }
    }
    catch(Exception e)
    {
      throw new IDNotFoundException("Could not resolve LSID: " + e.getMessage());
    }
  }

  /**
   * resolves an id to a class name
   * @param id the id to resolve
   * @return id of the class
   * @throws IDNotFoundException if the id cannot be resolved to a class
   */
  public ActorMetadata resolveIDToActor(KeplerLSID id)
    throws IDNotFoundException, InvalidMetadataException
  {
    return new ActorMetadata(resolveID(id));
  }

  /**
   * resolve the id to a class
   */
  public ActorMetadata resolveIDToActor(String id)
    throws IDNotFoundException, InvalidMetadataException
  {
    try
    {
      return resolveIDToActor(new KeplerLSID(id));
    }
    catch(LSIDException lside)
    {
      throw new IDNotFoundException(lside.getMessage());
    }
  }

  /**
   * returns true if the passed id is resolvable by the implemented id resolver
   * @param id
   * @return boolean
   */
  public boolean isResolvable(String id)
  {
    if(id.substring(0, 8).equals("urn:lsid"))
    {
      return true;
    }
    return false;
  }
}