/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.io.FileReader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;

import org.xml.sax.InputSource;
import org.xmlmiddleware.schemas.dtds.Attribute;
import org.xmlmiddleware.schemas.dtds.DTD;
import org.xmlmiddleware.schemas.dtds.DTDParser;
import org.xmlmiddleware.schemas.dtds.ElementType;


/**
 * The XMLSchemaParser is to store all subelements of each element in a dtd file
 *  into a  hashtable and attributes in another hashtable, then used by
 * SchemaTree to create an SchemaTree using insertRoot() and insertChildren().
 * The XMLSchemaParser was implemented in the same way. However this is not a
 * good way for the job at all. The XMLSchemaImporter was developed later, which
 * creates a SchemaTree directly from an input XML Schema file.
 *
 * @author Guilian Wang
 */
public class MyDTDParser {
    String schemaRootTag; // the tag of the Schema root element
    ElementType schemaRootElem; // the root element of DTD

    // hash table to hold the children of each valid node in the schema tree
    Hashtable elemHash = new Hashtable();
    Hashtable attrHash = new Hashtable();

    public MyDTDParser(String dtdFile) {
        try {
            InputSource src = new InputSource(new FileReader(dtdFile));

            /* For reading file from jar file
            InputStream is = this.getClass().getResourceAsStream(filePath);

            if (is == null)
                 if(SCIA.debug_on) System.err.println("InputStream = null");
            else
                if(SCIA.debug_on) System.err.println("InputStream of " + dtdFile + " = " +
                                   is.available());
            InputSource src = new InputSource(is);
            */
            DTDParser parser = new DTDParser();
            DTD dtd = parser.parseExternalSubset(src, null);
            processElementTypes(dtd, elemHash, attrHash);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static void main(String[] args) {
        /*
        if (args.length != 1)
            {
                if(SCIA.debug_on) System.err.println("Syntax: java MyDTDParser <dtd-file>");
                return;
         }
        */
        MyDTDParser dtdP = new MyDTDParser("book1.dtd");

        if (SCIA.debug_on) {
            System.err.println("schemaRoot = " + dtdP.schemaRootTag);
        }

        Enumeration e = dtdP.elemHash.keys();

        if (SCIA.debug_on) {
            System.err.println("elemHash = ");
        }

        while (e.hasMoreElements()) {
            String key = (String) e.nextElement();

            if (SCIA.debug_on) {
                System.err.print("   " + key + "= [");
            }

            List elems = (List) dtdP.elemHash.get(key);

            for (int i = 0; i < elems.size(); i++) {
                if (SCIA.debug_on) {
                    System.err.print(((ElementType) elems.get(i)).name.getLocalName() +
                        "  ");
                }
            }

            if (SCIA.debug_on) {
                System.err.println("]\n");
            }
        }

        if (SCIA.debug_on) {
            System.err.println("attrHash = ");
        }

        e = dtdP.attrHash.keys();

        while (e.hasMoreElements()) {
            String key = (String) e.nextElement();

            if (SCIA.debug_on) {
                System.err.print("   " + key + "= [");
            }

            List attrs = (List) dtdP.attrHash.get(key);

            for (int i = 0; i < attrs.size(); i++) {
                if (SCIA.debug_on) {
                    System.err.print(((Attribute) attrs.get(i)).name.getLocalName() +
                        "  ");
                }
            }

            if (SCIA.debug_on) {
                System.err.println("]\n");
            }
        }
    }

    private void processElementTypes(DTD dtd, Hashtable elemHash,
        Hashtable attrHash) throws Exception {
        Enumeration e;
        e = dtd.elementTypes.elements();

        while (e.hasMoreElements()) {
            processElementType((ElementType) e.nextElement(), elemHash, attrHash);
        }
    }

    private void processElementType(ElementType elementType,
        Hashtable elemHash, Hashtable attrHash) throws Exception {
        String tag = elementType.name.getLocalName();

        //check if the element is the dtd root
        if (elementType.parents.isEmpty() && !elementType.children.isEmpty()) {
            /* sometimes, there maybe some extra element definitions in
               the input dtd file, so take the first satisfied one as the
               root. But this is not a solution at all, because it seems
               that the order is not the same as the appearance in the dtd
               file. So dtd files have to be cleaned, to make sure there is
               no element that has children but is not referenced in any
               other element type definition.
            */
            if ((schemaRootTag == null) && (schemaRootElem == null)) {
                schemaRootTag = tag;
                schemaRootElem = elementType;
            }
        }

        // Check if the element is a parent. If not, return and don't
        // process it now. Instead, we will process it when we encounter it in
        // each of its parents.
        if (!isParent(elementType)) {
            return;
        }

        // Process the attributes hashtable, change it into a list, then add
        // it as an entry in attrHash, key = elementType.name.getLocalName()
        if (!elementType.attributes.isEmpty()) {
            List attrs = new ArrayList(elementType.attributes.values());
            attrHash.put(tag, attrs);
        }

        // Process the content, add the list of child elements into elemHash.
        List childElems = new ArrayList(elementType.children.values());
        elemHash.put(tag, childElems);

        /********
                 switch (elementType.contentType){
                 case ElementType.CONTENT_ANY:

                 case ElementType.CONTENT_MIXED:
                 throw new Exception("Can't process element types with mixed " +
                 "or ANY content: " + elementType.name.getUniversalName());

                 case ElementType.CONTENT_ELEMENT:
                 processElementContent(elementType.content, elementType.children);
                 break;

                 case ElementType.CONTENT_PCDATA:
                 processPCDATAContent(elementType);
                 break;

                 case ElementType.CONTENT_EMPTY:
                 // No content to process.
                 break;
                 }
        *******/
    }

    private boolean isParent(ElementType elementType) {
        // If an element type has any attributes or child elements, it is
        // treated as a parent. Otherwise, it is treated as a property.
        // BUG! This code actually misses a special case. If an element type
        // has no children, no attributes, and no parents, it needs to be
        // treated as a class. However, the corresponding XML document is:
        //
        //    <?xml version="1.0"?>
        //    <!DOCTYPE [<!ELEMENT foo EMPTY>]>
        //    <foo/>
        //
        // which really isn't worth worrying about...
        return (!elementType.children.isEmpty() ||
        !elementType.attributes.isEmpty());
    }
}
