/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import org.semanticweb.owl.model.OWLException;
import org.semanticweb.owl.model.OWLOntology;
import org.semanticweb.owl.model.helper.OntologyHelper;

import java.net.URI;
import java.net.URISyntaxException;

/*
 * OWLMetadata.java
 * Created on Jul 26, 2004
 *
 */
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Set;


/**
 * OWLMetadata.java
 * @author Rich
 *
 */
public class OWLMetadata {
    URI baseURI;
    List namespaces = new ArrayList();
    Hashtable uriHash = new Hashtable(); // hash uri strings and prefix strings
    Hashtable prefixHash = new Hashtable(); // hash prefix strings and uri strings
    Hashtable known = new Hashtable(); // hash known prefixes and uri strings
    Hashtable knownURI = new Hashtable(); // hash known uri and prefixes

    public OWLMetadata() {
        addKnown("xsd", "http://www.w3.org/2001/XMLSchema#");
    }

    public OWLMetadata(OWLOntology ontology, Hashtable prefixMap) {
        String uri = "http://a.com/ontology";

        if (ontology != null) {
            try {
                uri = ontology.getLogicalURI().toString();
            } catch (OWLException e) {
                e.printStackTrace();
            }
        }

        try {
            setBaseURI(new URI(uri));
        } catch (URISyntaxException e) {
            e.printStackTrace();
        }

        Enumeration keys = prefixMap.keys();
        Enumeration elem = prefixMap.elements();
        Set importURI = new HashSet();

        try {
            Set includeOntologies = OntologyHelper.importClosure(ontology);

            for (Iterator ontit = includeOntologies.iterator();
                    ontit.hasNext();) {
                OWLOntology onto = (OWLOntology) ontit.next();
                importURI.add(onto.getLogicalURI().toString() + "#");
            }
        } catch (OWLException e) {
            e.printStackTrace();
        }

        while (keys.hasMoreElements()) {
            String prefix = (String) keys.nextElement();
            uri = (String) elem.nextElement();

            boolean imported = !uri.toString().equals(getBaseURI().toString() +
                    "#") && importURI.contains(uri);
            addNamespace(uri, prefix, imported);
        }
    }

    private void addKnown(String prefix, String uri) {
        known.put(prefix, uri);
        knownURI.put(uri, prefix);
    }

    /**
     * Method getBaseURI
     * @return
     */
    public URI getBaseURI() {
        return baseURI;
    }

    /**
     * Method setBaseURI
     * @param uri
     */
    public void setBaseURI(URI uri) {
        String s;

        if (baseURI != null) {
            s = baseURI.toString() + "#";
            prefixHash.remove(getPrefixOfURI(s));
            uriHash.remove(s);
        }

        baseURI = uri;
        s = uri.toString() + "#";
        uriHash.put(s, "");
        prefixHash.put("", s);
    }

    public void addNamespace(String uri, String prefix, boolean imported) {
        //TODO check it's not already there
        if (!uriHash.containsKey(uri)) {
            uriHash.put(uri, prefix);
        }

        if (!prefixHash.containsKey(prefix)) {
            prefixHash.put(prefix, uri);
        }

        namespaces.add(new NameSpace(uri, prefix, imported));
    }

    public void changeNamespace(int index, String uri, String prefix) {
        NameSpace ns = (NameSpace) namespaces.get(index);
        prefixHash.remove(getPrefixOfURI(uri));
        uriHash.remove(uri);

        //TODO  check it's not already there
        if (!uriHash.containsKey(uri)) {
            uriHash.put(uri, prefix);
        }

        prefixHash.put(prefix, uri);
        ns.setUri(uri);
        ns.setPrefix(prefix);
    }

    public void deleteNamespace(int index) {
        NameSpace ns = (NameSpace) namespaces.get(index);
        String s = ns.getUri();
        String p = getPrefixOfURI(s);
        uriHash.remove(s);
        prefixHash.remove(p);
        namespaces.remove(index);
    }

    List getNamespaces() {
        return namespaces;
    }

    /**
     * Method getURIOfPrefix
     * If the prefix is known, return the corresponding uri.
     * Otherwise return the string that was passed in.
     * @param prefix
     * @return
     */
    public String getURIOfPrefix(String prefix) {
        String uri;

        if (prefix == null) {
            prefix = "";
        }

        uri = (String) prefixHash.get(prefix);

        if (uri == null) {
            uri = (String) known.get(prefix);
        }

        return (uri == null) ? prefix : uri;
    }

    private String getURIFromPrefixHash(String prefix) {
        return (String) prefixHash.get(prefix);
    }

    public String getPrefixOfFullURI(URI uri) {
        String ssp = getURISSP(uri);
        String prefix = (String) uriHash.get(ssp);

        if (prefix == null) {
            if ((prefix = (String) knownURI.get(ssp)) == null) {
                prefix = ssp;
            }
        }

        return prefix;
    }

    public String getNamespacePrefix(URI uri) {
        String pp = getPrefixOfFullURI(uri);

        if (pp.equalsIgnoreCase("")) {
            return pp;
        } else {
            return pp + ":";
        }
    }

    public String getPrefixOfURI(String uri) {
        return (String) uriHash.get(uri);
    }

    private String getURISSP(URI uri) {
        String ssp = null;

        try {
            if (uri.getFragment() != null) {
                /* It's of the form http://xyz/path#frag */
                ssp = new URI(uri.getScheme(), uri.getSchemeSpecificPart(), null).toString();
            }
        } catch (URISyntaxException ex) {
        }

        if (ssp == null) {
            ssp = uri.toString();
        }

        if (!ssp.endsWith("#")) {
            ssp = ssp + "#";
        }

        return ssp;
    }

    public String shortForm(URI uri) {
        if (uri == null) {
            return "_";
        }

        try {
            if (uri.getFragment() == null) {
                /* It's not of the form http://xyz/path#frag */
                if (SCIA.debug_on) {
                    System.err.println("not regular form");
                }

                return uri.toString();
            }

            /* It's of the form http://xyz/path#frag */
            String ssp = new URI(uri.getScheme(), uri.getSchemeSpecificPart(),
                    null).toString();

            //System.out.println("ssp=" + ssp);	  
            if (!ssp.endsWith("#")) {
                ssp = ssp + "#";
            }

            if (knownURI.contains(ssp)) // common schemas that are displayed without a prefix
             {
                if (SCIA.debug_on) {
                    System.err.println("knownURI contains ssp");
                }

                return uri.getFragment();
            }

            String prefix = getPrefixOfURI(ssp);

            if (prefix == null) {
                if (SCIA.debug_on) {
                    System.err.println("prefix = null");
                }

                return uri.toString();
            } else if (prefix.length() == 0) {
                //System.out.println("prefix length = 0");	  
                return uri.getFragment();
            } else {
                if (SCIA.debug_on) {
                    System.err.println("prefix of ssp = " + prefix);
                }

                return prefix + ":" + uri.getFragment();
            }
        } catch (URISyntaxException ex) {
        }

        return uri.toString();
    }

    public URI uriFromShortForm(String s) {
        StringBuffer sb = new StringBuffer();
        String[] parts = s.split(":", 2);

        if (parts.length == 2) {
            sb.append(getURIOfPrefix(parts[0]));
            sb.append(parts[1]);
            s = sb.toString();
        } else {
            sb.append(baseURI.toString());
            sb.append("#");
            sb.append(parts[0]);
            s = sb.toString();
        }

        try {
            return new URI(s);
        } catch (URISyntaxException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        return null;
    }

    public String fragmentFromShortForm(String s) {
        String[] parts = s.split(":", 2);

        if (parts.length == 2) {
            return parts[1];
        }

        return s;
    }

    public void setDefaults() {
        namespaces.clear();

        try {
            setBaseURI(new URI("http://a.com/ontology"));
            addNamespace("http://www.w3.org/2002/07/owl#", "owl", false);
            addNamespace("http://www.w3.org/1999/02/22-rdf-syntax-ns#", "rdf",
                false);
            addNamespace("http://www.w3.org/2000/01/rdf-schema#", "rdfs", false);
        } catch (URISyntaxException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    public String getTypeShortForm(String name) {
        if (name.equalsIgnoreCase("Literal")) {
            return "rdfs:" + name;
        }

        return "xsd:" + name;
    }

    public class NameSpace {
        String uri;
        String prefix;
        boolean imported;

        NameSpace(String u, String pre, boolean imp) {
            uri = u;
            prefix = pre;
            imported = imp;
        }

        /**
         * Method isImported
         * @return
         */
        public boolean isImported() {
            return imported;
        }

        /**
         * Method getPrefix
         * @return
         */
        public String getPrefix() {
            return prefix;
        }

        /**
         * Method getUri
         * @return
         */
        public String getUri() {
            return uri;
        }

        /**
         * Method setImported
         * @param b
         */
        public void setImported(boolean b) {
            imported = b;
        }

        /**
         * Method setPrefix
         * @param string
         */
        public void setPrefix(String string) {
            prefix = string;
        }

        /**
         * Method setUri
         * @param uri
         */
        public void setUri(String uri) {
            this.uri = uri;
        }
    }
}
