/**
 *    '$RCSfile: ChannelSuggestionDialog.java,v $'
 *
 *     '$Author: jwang $'
 *       '$Date: 2005/12/22 01:32:17 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler-project.org
 *
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.scia.gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.ArrayList;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.kepler.scia.ActorsImporter;
import org.kepler.scia.SCIA;
import org.kepler.scia.SchemaTree;
import org.kepler.scia.TNode;
import org.kepler.sms.SemanticType;

import ptolemy.actor.Actor;
import ptolemy.actor.AtomicActor;
import ptolemy.actor.CompositeActor;
import ptolemy.actor.IOPort;
import ptolemy.actor.IORelation;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedCompositeActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.actor.TypedIORelation;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.ComponentPort;
import ptolemy.kernel.ComponentRelation;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Entity;
import ptolemy.moml.MoMLChangeRequest;
import ptolemy.vergil.actor.ActorGraphFrame;
import ptolemy.vergil.actor.ActorGraphTableau;
import ptolemy.vergil.basic.BasicGraphFrame;
import diva.graph.GraphController;
import diva.graph.GraphModel;
import diva.graph.GraphPane;
import diva.graph.JGraph;

/**
 * This dialog provides a simple interface for selecting the source actors 
 * and the target actors for SCIA to match them and suggest connection 
 * channels between the output ports of the source actors to the input ports 
 * of the target actors.
 * @author Guilian Wang
 */
public class ChannelSuggestionDialog extends JDialog {

    /* PRIVATE MEMBERS */

    private int[] _selectedSourceActorIndices;
    private int[] _selectedTargetActorIndices;
    private java.util.List _selectedSourceActors;
    private java.util.List _selectedTargetActors;;
    private SchemaTree _targetTree; 	
    private boolean _sourceSelected = false;
    private boolean _targetSelected = false;
    private JList _sActorsList;
    private JList _tActorsList;
    private java.util.List _actors;

    private JPanel _pane = null;  // overall pane for the dialog

    private JButton _launchMatcherBtn =   // button to launch matcher
	new JButton("Launch Matcher"); 

    private JButton _applySuggestionBtn =   // button to apply matches into workflow
	new JButton("Apply Matches"); 

    private JButton _closeBtn = new JButton("Close");  // button to close dialog

    private Frame _owner;   // the owner of this dialog
    private Entity _entity; // the entity calling the dialog

    /**
     * Default construct
     * @param owner The frame that called the dialog
     * @param entity The entity whose actors are being matched.
     */
    public ChannelSuggestionDialog(Frame owner, Entity entity) {
	super(owner);
	_owner = owner;
	_entity = entity;
	this.setTitle("Actor Matcher");
	_initializeDialog();
	this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

	try {
	    if(entity instanceof CompositeEntity) {
		System.out.println("Is compositeEntity");
	    }
	} catch(Exception e) {}
    }

    
    /**
     * This method initializes the dialog
     */
    private void _initializeDialog() {
	_pane = new JPanel();
	_pane.setLayout(new BoxLayout(_pane, BoxLayout.Y_AXIS));
	_pane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));
	_pane.add(_createActorsPane());
	_pane.add(Box.createRigidArea(new Dimension(0, 10)));	
	_pane.add(_createButtons());

	this.setSize(750,500);
	this.setResizable(false);
	this.setContentPane(_pane);
    }

    /**
     * Creates an actor pane for the user to select source actors and select 
     * targetv actors from the list of available actors on the canvas to match
     */
    private JPanel _createActorsPane() {

	JPanel actorsPane = new JPanel();
	actorsPane.setLayout(new BoxLayout(actorsPane, BoxLayout.X_AXIS));

	//actors in the canvas
	_actors = _getActors();
	String[] actorNames = new String[_actors.size()];
	for (int i = 0; i < _actors.size(); i++) {
	    actorNames[i] = ((Actor)_actors.get(i)).getName();
	    System.out.println("actorNames[" + i + "] = " + actorNames[i]);
	}

	//source actors list
	_sActorsList = new JList(actorNames);

	ListSelectionListener lSL = new ListSelectionListener () {
		public void valueChanged(ListSelectionEvent e) {
		    if (e.getValueIsAdjusting() == false) {

			if (e.getSource() == _sActorsList) {
			    if (_sActorsList.getSelectedIndex() == -1) {
				//No selection, disable launch matcher button.
				_launchMatcherBtn.setEnabled(false);		    
			    } else {
				_sourceSelected = true;
				_selectedSourceActorIndices = 
				    _sActorsList.getSelectedIndices();
			    } 
			} else if (e.getSource() == _tActorsList) {
			    if (_tActorsList.getSelectedIndex() == -1) {
				//No selection, disable launch matcher button.
				_launchMatcherBtn.setEnabled(false);
		    
			    } else {
				_targetSelected = true;
				_selectedTargetActorIndices = 
				    _tActorsList.getSelectedIndices();
			    }
			}

			//both source and target selected, 
			//enable launch matcher button.
			if (_sourceSelected && _targetSelected) {
			    _launchMatcherBtn.setEnabled(true);
			}
		    }
		}
	    };

	_sActorsList.addListSelectionListener(lSL);
	_sActorsList.setVisibleRowCount(5);

	//put _sActorsList into a scroll panel
	JScrollPane sourcePane = new JScrollPane(_sActorsList);
	sourcePane.setBorder(_createTitledBorder("Select source actors"));
 	//sourcePane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

	//target actors list
	_tActorsList = new JList(actorNames);
	//_tActorsList = new JList(_getActors().toArray());
	_tActorsList.addListSelectionListener(lSL);
	_tActorsList.setVisibleRowCount(5);

	//put _tActorsList into a scroll panel
	JScrollPane targetPane = new JScrollPane(_tActorsList);
	targetPane.setBorder(_createTitledBorder("Select target actors"));
 	//targetPane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

	actorsPane.add(sourcePane);
	actorsPane.add(Box.createRigidArea(new Dimension(10,0)));	
	actorsPane.add(targetPane);

	return actorsPane;
    }

    /** 
     * @return A list of actors 
     */
    private java.util.List _getActors() {
	java.util.List actors = new ArrayList();
	
	if(_entity == null || !(_entity instanceof CompositeEntity)) {
	    return null;
	}
	
	//it only make sense for compositeEntity
	if (!(_entity instanceof CompositeEntity)) {
	    System.out.println("not CompositeEntity");
	    return null;
	}
	CompositeEntity ce = (CompositeEntity)_entity;	

	// get the entity list on the canvas
	java.util.List entities = ce.entityList();
	for (int i = 0; i < entities.size(); i++) {
	    Entity e = (Entity)entities.get(i);
	    System.out.println("entity " + i + " = " + e.getName());
	    if((e instanceof AtomicActor) || (e instanceof CompositeActor)) {
		System.out.println("actor " + i + " = " + e.getName());
		actors.add((Actor)e);
	    }
	}
	return actors;
    }

    /**
     * Create the bottom buttons
     */
    private JPanel _createButtons() {
	JPanel pane = new JPanel();
	pane.setLayout(new BoxLayout(pane, BoxLayout.LINE_AXIS));
 	pane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));
	pane.add(Box.createHorizontalGlue());
	pane.add(_launchMatcherBtn);
	pane.add(Box.createRigidArea(new Dimension(10, 0)));
	pane.add(_applySuggestionBtn);
	pane.add(Box.createRigidArea(new Dimension(10, 0)));
	pane.add(_closeBtn);

	// init buttons
	_launchMatcherBtn.setMnemonic(KeyEvent.VK_A);
	_launchMatcherBtn.setActionCommand("launchMatcher");
	_launchMatcherBtn.setToolTipText
            ("Launch actor matcher for suggesting connections");
	_launchMatcherBtn.addActionListener(_buttonListener);
	_launchMatcherBtn.setEnabled(false);

	_applySuggestionBtn.setMnemonic(KeyEvent.VK_A);
	_applySuggestionBtn.setActionCommand("applyMatches");
	_applySuggestionBtn.setToolTipText("Apply matches to the workflow");
	_applySuggestionBtn.addActionListener(_buttonListener);

	_closeBtn.setActionCommand("close");
	_closeBtn.setToolTipText("Close channel suggestion dialog");
	_closeBtn.addActionListener(_buttonListener);
	
	return pane;
    }


    /**
     * Create a titled border with a blue font
     */
    private TitledBorder _createTitledBorder(String title) {
	TitledBorder border = BorderFactory.createTitledBorder(" " + title + " ");
	border.setTitleColor(new Color(0, 10, 230));
	return border;
    }

    /**
     * Launch SCIA for matching the selected source actors and target actors
     */
    private void _launchSCIA() {

	_selectedSourceActors = new ArrayList();
	_selectedTargetActors = new ArrayList();
	for (int i = 0; i < _selectedSourceActorIndices.length; i++) {
	    _selectedSourceActors.add(_actors.get(_selectedSourceActorIndices[i]));
	}

	for (int i = 0; i < _selectedTargetActorIndices.length; i++) {
	    _selectedTargetActors.add(_actors.get(_selectedTargetActorIndices[i]));
	}

	ActorsImporter ai = new ActorsImporter(_selectedSourceActors, false);
	SchemaTree sourceTree = new SchemaTree(ai.buildSchemaTree());

	ai = new ActorsImporter(_selectedTargetActors, true);
	_targetTree = new SchemaTree(ai.buildSchemaTree());

	//to do: guess domain from the paths of the actors
	// launch SCIA GUI
	SCIA.matchActors(sourceTree, _targetTree);
    }

    /**
     * anonymous class to handle button events
     */
    private ActionListener _buttonListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		if(e.getActionCommand().equals("launchMatcher")) {
		    _launchSCIA();
		} else if (e.getActionCommand().equals("applyMatches")) {
		    _applySuggestion(_targetTree);
		} else if(e.getActionCommand().equals("close")) {
		    //_unHighlightEdges();
		    dispose();
		}
	    }
	};

   /**
     * Applies the SCIA suggested connections between the ports of the 
     * selected source actors and target actors, i.e., add the matches as 
     * relations into the workflow and display them in the workspace.
     *
     * @param SchemaTree The SchemaTree that represents the target actors and 
     * holds the generated matches
     */
    private void _applySuggestion(SchemaTree tree) {

	if(!(_owner instanceof ActorGraphFrame)) {
	    return;
	}

	if (tree == null || tree.rootNode == null || 
	    tree.rootNode.children == null ||
	    tree.rootNode.children.isEmpty()) {
	    return;
	}

	long relNum = 0; // relation number

	// get the tableau
	ActorGraphFrame tmpGraphFrame = (ActorGraphFrame)_owner;
	ActorGraphTableau tmpTableau = 
	    (ActorGraphTableau)tmpGraphFrame.getTableau();

	// get the controller
	BasicGraphFrame tmpParent = (BasicGraphFrame)tmpTableau.getFrame();
	JGraph tmpJGraph = tmpParent.getJGraph();
	GraphPane tmpGraphPane = tmpJGraph.getGraphPane();
	GraphController controller = 
	    (GraphController)tmpGraphPane.getGraphController();

	// get the graph model
	GraphModel graphModel = controller.getGraphModel();

	for (int i = 0; i < tree.rootNode.children.size(); i++) {
	    Actor targetActor = null;
	    TNode actorNode = (TNode)tree.rootNode.children.get(i);
	    if (actorNode.hasDecendentMatched()) {

		//identify the corresponding target actor
		for (int j = 0; j < _selectedTargetActors.size(); j++) {
		    Actor actor = (Actor)_selectedTargetActors.get(j);
		    if (actor.getName().equals(actorNode.getTag())) {
			targetActor = actor; 
			System.err.println("targetActor = " + targetActor.getName());
			break;
		    }
		}

		// process each port of the target actor
		for (int k = 0; k < actorNode.children.size(); k++) {
		    Actor sourceActor = null;
		    IOPort targetPort = null;
		    IOPort sourcePort = null;
		    //Node sourceNode = null;
		    //Node targetNode = null;
		    TNode portNode = (TNode)actorNode.children.get(k);

		    //identify the target port
		    java.util.List ports = targetActor.inputPortList();
		    for (int h = 0; h < ports.size(); h++) {		
			IOPort port = (IOPort)ports.get(h);	
			if (port.getName().equals(portNode.getTag())) {
			    targetPort = port;
			    System.err.println("targetPort = " + targetPort.getName());
			    break;
			}
		    }
		    

		    if (portNode.fromComb != null && 
			portNode.fromComb.tnode != null &&
			portNode.fromComb != null &&
			portNode.parent.fromComb != null) {

			String matchPort = portNode.fromComb.tnode.getTag();
			String matchActor = portNode.fromComb.tnode.parent.getTag();

			//identify the matchActor and port
			for (int j = 0; j < _selectedSourceActors.size(); j++) {
			    Actor actor = (Actor)_selectedSourceActors.get(j);
			    if (actor.getName().equals(matchActor)) {
				sourceActor = actor; 		
				System.err.println("sourceActor = " + 
						   sourceActor.getName());		
				break;
			    }
			}
			
			//identify the source port
			ports = sourceActor.outputPortList();
			for (int h = 0; h < ports.size(); h++) {		
			    IOPort port = (IOPort)ports.get(h);	
			    System.err.println("source actor port = " + port.getName());

			    if (port.getName().equals(matchPort)) {
				sourcePort = port;
				System.err.println("sourcePort = " + 
						   sourcePort.getName());
				break;
			    }
			}

			// connect sourcePort to targetPort
			try {
			    ComponentRelation rel = ((CompositeEntity)_entity).
				connect((ComponentPort)sourcePort, 
					(ComponentPort)targetPort);
			    //rel.setContainer((CompositeEntity)_entity);
			    String moml = rel.exportMoML();
			    _entity.requestChange(new MoMLChangeRequest
						  (this, _entity, moml));
			} catch (Exception e) {
			    e.printStackTrace();
			}
			

			/* search for the first available relation number 
			while(true) {
			    Relation rel = ((CompositeEntity)_entity).
				getRelation("relation" + relNum);
			    if (rel == null) {
				break;
			    }
			    else relNum++;
			}

			// add relation
			StringBuffer rels = new StringBuffer();
			rels.append("    <relation name=\"relation"+ relNum + 
				    "\"class=\"ptolemy.actor.TypedIORelation\">\n");
			rels.append("    </relation>\n");
       
			_entity.requestChange(new MoMLChangeRequest(this, _entity, 
								  rels.toString()));
       
			// add sourceActor's sourcePort to the relation.
			String request = "<link port=\"" + sourceActor.getName() + 
			    "." + sourcePort.getName() + "\"relation=\"relation" + 
			    relNum + "\"/>";
			_entity.requestChange(new MoMLChangeRequest
					    (this, _entity, request));
       
			// add targetActor's targetPort to the relation.
			request = "<link port=\"" + targetActor.getName() + 
			    "." + targetPort.getName() + "\"relation=\"relation" + 
			    relNum + "\"/>";
			_entity.requestChange(new MoMLChangeRequest
					      (this, _entity, request));
       
			relNum++;
			*/
		    }
		}		
	    }	    	    
	}

	//locatableNodeDragInteracter in vergial.basic
	//frame??
	//frame.changeExecuted(null);
	//_owner.paintAll(_owner.getGraphics());

    }
   



    /**
     * insert adapters into workflow ...
     */
    private void _insertAdapters() {
	JOptionPane.showMessageDialog(this, "This function is not implemented yet.");
    }


    /**
     * Main method for testing the dialog.
     * 
     * @param args the arguments to the program
     */
    public static void main(String[] args) {
	try {
	    // the workflow
	    TypedCompositeActor swf = new TypedCompositeActor();

	    // the two actors
	    //CompositeActor a1 = new TypedCompositeActor(swf, "Actor1");
	    //CompositeActor a2 = new TypedCompositeActor(swf, "Actor2");
	    TypedAtomicActor a1 = new TypedAtomicActor(swf, "Actor1");
	    TypedAtomicActor a2 = new TypedAtomicActor(swf, "Actor2");	    

	    // a port with <unknown, error> data/semantic type
	    IOPort p1_a1 = new TypedIOPort(a1, "p1", false, true);
	    IOPort p1_a2 = new TypedIOPort(a2, "p1", true, false);
	    IORelation r1 = new TypedIORelation(swf, "r1");    
	    p1_a1.link(r1);
	    p1_a2.link(r1);
	    SemanticType s1_p1_a1 = new SemanticType(p1_a1, "semType0");
	    s1_p1_a1.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#BiomassMeasurement");
	    SemanticType s1_p1_a2 = new SemanticType(p1_a2, "semType0");
	    s1_p1_a2.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#Biomass");	    

	    // a port with <error, unknown> data/semantic type
	    TypedIOPort p2_a1 = new TypedIOPort(a1, "p2", false, true);
	    TypedIOPort p2_a2 = new TypedIOPort(a2, "p2", true, false);
	    p2_a1.setTypeEquals(new ArrayType(BaseType.DOUBLE));
	    p2_a2.setTypeEquals(BaseType.DOUBLE);
	    IORelation r2 = new TypedIORelation(swf, "r2");    
	    p2_a1.link(r2);
	    p2_a2.link(r2);

	    // a port with <safe, safe> data/semantic type
	    TypedIOPort p3_a1 = new TypedIOPort(a1, "p3", false, true);
	    TypedIOPort p3_a2 = new TypedIOPort(a2, "p3", true, false);
	    p3_a1.setTypeEquals(new ArrayType(BaseType.DOUBLE));
	    p3_a2.setTypeEquals(BaseType.GENERAL);
	    IORelation r3 = new TypedIORelation(swf, "r3");    
	    p3_a1.link(r3);
	    p3_a2.link(r3);
	    SemanticType s2_a1 = new SemanticType(p3_a1, "semType0");
	    s2_a1.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#SpeciesBiomassMeasurement");
	    SemanticType s2_a2 = new SemanticType(p3_a2, "semType0");
	    s2_a2.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#BiomassMeasurement");	    

	    // a relation with multiple channels (p4 error, error with p5)
	    TypedIOPort p4_a1 = new TypedIOPort(a1, "p4", false, true);
	    TypedIOPort p5_a1 = new TypedIOPort(a1, "p5", false, true);
	    TypedIOPort p4_a2 = new TypedIOPort(a2, "p4", true, false);	    
	    TypedIOPort p5_a2 = new TypedIOPort(a2, "p5", true, false);	    
	    p4_a1.setTypeEquals(new ArrayType(BaseType.DOUBLE));
	    p5_a1.setTypeEquals(BaseType.STRING);	    
	    p4_a2.setTypeEquals(new ArrayType(BaseType.DOUBLE));
	    p5_a2.setTypeEquals(new ArrayType(BaseType.INT));	    
	    IORelation r4 = new TypedIORelation(swf, "r4");    
	    p4_a1.link(r4);
	    p4_a2.link(r4);
	    p5_a1.link(r4);
	    p5_a2.link(r4);
	    SemanticType s3_a1 = new SemanticType(p4_a1, "semType0");
	    s3_a1.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#CoverArea");
	    SemanticType s3_a2 = new SemanticType(p4_a2, "semType0");
	    s3_a2.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#Biomass");	    
	    SemanticType s4_a1 = new SemanticType(p5_a1, "semType0");
	    s4_a1.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#Species");
	    SemanticType s4_a2 = new SemanticType(p5_a2, "semType0");
	    s4_a2.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#Species");


	    // what do we do with multiple semantic types?  (There has to be a "cross-wise" match)
	    // a port with <safe, safe> data/semantic type
	    TypedIOPort p6_a1 = new TypedIOPort(a1, "p6", false, true);
	    TypedIOPort p6_a2 = new TypedIOPort(a2, "p6", true, false);
	    //p6_a1.setTypeEquals(new ArrayType(BaseType.DOUBLE));
	    //p3_a2.setTypeEquals(BaseType.GENERAL);
	    IORelation r5 = new TypedIORelation(swf, "r5");    
	    p6_a1.link(r5);
	    p6_a2.link(r5);
	    SemanticType s5_a1 = new SemanticType(p6_a1, "semType0");
	    s5_a1.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#SpeciesBiomassMeasurement");
	    SemanticType s5_a2 = new SemanticType(p6_a2, "semType0");
	    s5_a2.setExpression("urn:lsid:lsid.ecoinformatics.org:onto:3:1#BiomassMeasurement");	    	    

	    // windows look and feel
	    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
	    ChannelSuggestionDialog matcher = new ChannelSuggestionDialog(null, swf);
	    matcher.setVisible(true);
	} catch(Exception e) {
	    e.printStackTrace();
	}
    }

}
