/* Base class for actors that process collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

 package org.nddp;
  
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.nddp.exceptions.CollectionException;
import org.nddp.tokens.ClosingDelimiterToken;
import org.nddp.tokens.CollectionToken;
import org.nddp.tokens.DataToken;
import org.nddp.tokens.DomainObjectToken;
import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.LoopTerminationToken;
import org.nddp.tokens.MetadataToken;
import org.nddp.tokens.OpeningDelimiterToken;
import org.nddp.tokens.VariableToken;

import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.Nameable;
import ptolemy.kernel.util.NamedObj;
 
/**
 * Base class for actors that process collections. The class simplifies the 
 * writing of such actors by providing the following services:
 * <ul>
 * <li>Provides a fire method that calls a handler method for each data token 
 * received.
 * <li>Provides implementations of the non-data handler methods that call 
 * corresponding handlers when a new start of a new collection arrives, when
 * the end of a collection arrives, when a metadata token arrives, etc.
 * <li>Creates collection managers for each incoming collection and passes 
 * references to the collection event handlers.
 * <li>Allows subclasses to specify whether to process and whether to forward
 * on the collections that are received by the actor.  The subclass is not
 * informed of arriving tokens and subcollections part of a collection that 
 * the subclass has chosen not to process.
 * <li>Provides a parameter for specifying types of collections and data that 
 * should be processed.  All other collections and data are passed through 
 * transparently.  
 * </ul>
 * 
 *  @author Timothy M. McPhillips
 */

	public class CollectionHandler implements CollectionIOPort.NondataHandler {

  	/** 
    	 * Constructs an instance of CollectionHandler
   	 * 
   	 * @param container The run-time container of this actor.
 	 * @param name The name of this actor.
  	 * @throws IllegalActionException 
    	 */
	public CollectionHandler(Coactor actor, CollectionPath collectionPath,
	        Map overridableParametersMap, Map overriddenParametersMap) 
        throws IllegalActionException {

    	    // call the superclass constructor
        	super();

        // store reference to collection actor
        _actor = actor;

        // store collection path
        _collectionPath = collectionPath;

        _overriddenParametersMap = overriddenParametersMap;
        
        _overridableParametersMap = overridableParametersMap;

        // we start below the upper collections to ignore if  the 
        // ignoreCollectionsAboveType parameter is the empty string; 
        // otherwise we are above it
        _belowUpperIgnoredTypes = 
            _collectionPath.ignoreCollectionsOutside() == Object.class;
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////
	
    	public static final TokenDisposition DISCARD_TOKEN = 
    	    new TokenDisposition();   
    	
    	public static final TokenDisposition FORWARD_TOKEN = 
    	    new TokenDisposition();
    	
    	public static final ExceptionDisposition DISCARD_EXCEPTION =
    	    new ExceptionDisposition();
    	
    	public static final ExceptionDisposition PROPAGATE_EXCEPTION =
    	    new ExceptionDisposition();
    	
    	public static final CollectionDisposition 
    		IGNORE_AND_DISCARD_COLLECTION = new CollectionDisposition(); 
    	
	public static final CollectionDisposition 
		IGNORE_AND_FORWARD_COLLECTION = new CollectionDisposition();
	
    	public static final CollectionDisposition 
    		PROCESS_AND_DISCARD_COLLECTION = new CollectionDisposition();
    	
	public static final CollectionDisposition 
		PROCESS_AND_FORWARD_COLLECTION = new CollectionDisposition();
	
//	public Parameter collectionPath;
	
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    /** 
     * The method called by the director when the actor is fired. The method
     * reads the next data token from the input port, passes the data token to
     * the handleData method, and then passes the data on via the current
     * collection manager if the collection has not been discarded and the 
     * handleData method did not return DO_NOT_SEND_DATA.
     * @throws IllegalActionException
     * @throws 
     */
    public void handleDataToken(Token token) throws IllegalActionException {
 
        // the block below is a kludge to allow collection actors to handle
        // data outside of collections.
        // TODO require child class to enable explicitly the processing of 
        // data outside of collections
        if (_collectionManager == null) {  
            	try {                
            	    _actor._handleData(null, token);
                
            } catch (CollectionException exception) {
                    
            		throw new IllegalActionException(
                		(Nameable)_actor, exception.getMessage());
            }
        		return;
        }
        
        // assert that if token is a CollectionToken then the token's
        // collection is in the current collection
	 	assert !(token instanceof CollectionToken) || 
	 		((CollectionToken)token).collection() == 
	 		    _collectionManager.collection():
		    "Token outside of collection: " +
		    "\n\t token = " + token + "\n\t token's collection = " + 
		    ((CollectionToken)token).collection() + 
		    "\n\t current collection = " + _collectionManager.collection() +
		    "\n\t actor = " + _actor;
        
	    	// if not discarding this collection and this is the first data token
	 	// for the collection, inform collection manager accordingly
        if (! _collectionManager.discarded() && 
                ! _collectionManager._firstDataReceived()) {
            
            _collectionManager._announceFirstData();
        }
	    
        _unwrappedDataToken = null;
        _unwrappedDomainObject = null;
        TokenDisposition dataDisposition = _actor.defaultDataDisposition();

    		if (! _collectionManager.ignored() && _typeMatchesTarget(token)) {
    
    		    try {
			    if (_unwrappedDataToken != null) {
			        
				dataDisposition = _actor._handleData(_collectionManager, 
				        _unwrappedDataToken);
				
			    } else {
			        
			        dataDisposition = _actor._handleDomainObject(_collectionManager, 
	                _unwrappedDomainObject);
			    }
			    
    		    } catch(CollectionException exception) {
    		        
				processCollectionException(exception);
    		    }
    		}
    		
		// pass the data on via the collection manager if collection is 
		// not discarded and the disposition of the data token is SEND_DATA
		if (! _collectionManager.discarded() && 
		        dataDisposition == FORWARD_TOKEN) {
		    
        		_collectionManager._passData(token);
    		}
    	}   

	/** 
    	 * Handles closing delimiter tokens for collection actors. Passes the
    	 * collection manager for the delimited collection to the 
    	 * handleCollectionEnd method if the collection is being processed.
    	 *  
    	 * @param token The closing delimiter token.
    	 * @return DO_NOT_SEND_TOKEN in all cases. The collection manager will
    	 * send the token along with the rest of the collection later if the 
    	 * collection is not discarded.
	 * @throws CollectionException
    	 */
	public void handleClosingDelimiter(ClosingDelimiterToken token) 
		throws IllegalActionException {

         // ensure that the closing delimiter is for the current collection  
	 	assert token.collection() == _collectionManager.collection():
		    "Closing delimiter for wrong collection: " +
		    "\n\t token = " + token +
		    "\n\t delimiter's collection = " + token.collection() +
		 	"\n\t current collection = " + _collectionManager.collection() + 	    
		    "\n\t actor = " + this;

	 	// call the collection-end handler if this delimited collection is not 
	 	// being ignored
	 	if (! _collectionManager.ignored() && 
 	        _collectionPath.exposeNonDataEvents()) {
			
	 	    try {
               _actor._handleCollectionEnd(_collectionManager);
            } catch (CollectionException exception) {
		       processCollectionException(exception);
            }
		}
	 	
		// close the collection if it is not being discarded
		if (! _collectionManager.discarded()) {
	    		_collectionManager.closeCollection();
		}
		
		// note that we are no longer below the highest collection type that 
		// should be processed if the closing delimiter is of the highest
		// collection being processed
		if (_belowUpperIgnoredTypes && _collectionManager.collection() == 
		    		_highestProcessedCollection) {
		    
		    // set the flag
			_belowUpperIgnoredTypes = false;
			
			// discard reference to highest collection processed
			_highestProcessedCollection = null;
		}
		
		// revert parameters overridden in this collection to previous values
		HashMap oldParameterValues = _collectionManager._oldParameterValues();
		
		for (Iterator i = oldParameterValues.keySet().iterator();
				i.hasNext(); ) {
		    
		    Parameter parameter = (Parameter)i.next();
		    Token previousValue = (Token)oldParameterValues.get(parameter);
            _actor._handleParameterChange(parameter, previousValue);
            _overriddenParametersMap.put(parameter, previousValue);
		}
    		
		// pop the parent collection manager off the stack and reinstate it
		// as the current collection manager
		_collectionManager = ((IncomingCollectionManager)
		        (_collectionManagerStack.pop()));
		
		// note that we are now above the lowest collection type that should 
		// be processed if closing delimiter is of the lowest collection being
		// processed
		if (! _aboveLowerIgnoredTypes && _collectionManager != null 
		        && _collectionManager.collection() == 
		            _lowestProcessedCollection) {
			
		    // set the flag
		    _aboveLowerIgnoredTypes = true;
		    
		    // discard reference to the lowest collection
			_lowestProcessedCollection = null;
		}
	}
   	   	
    public void handleExceptionToken(ExceptionToken token) 
    		throws IllegalActionException {

        // ensure that each BToken arrives between the opening and closing
        // delimiters of its collection  
	 	assert token.collection() == _collectionManager.collection():
		    "Token outside of collection: " +
		    "\n\t token = " + token + 
		    "\n\t token's collection = " + token.collection() +
		 	"\n\t current collection = " + _collectionManager.collection() + 	    
		    "\n\t actor = " + this;
	 	
	 	TokenDisposition exceptionDisposition = FORWARD_TOKEN;
	 	
	 	if (! _collectionManager.ignored() && 
 	        _collectionPath.exposeNonDataEvents()) {
	 	    
	 	    try {
                exceptionDisposition = _actor._handleExceptionToken(
                        _collectionManager, token);
            } catch (CollectionException exception) {
		       	processCollectionException(exception);
            }
	 	}
	 	
        if (exceptionDisposition == FORWARD_TOKEN) {
	 		_collectionManager._storeInputExceptionToken(token);
	 	}	
	} 


	/** 
    	 * Handles metadata tokens for collection actors. Passes the metadata 
    	 * token and the current collection manager to the handleMetadata
    	 * method if the collection is being processed.
    	 *  
    	 * @param token The metadat delimiter token.
    	 * @return DO_NOT_SEND_TOKEN in all cases. The collection manager will
    	 * send the token along with the rest of the collection later if the
    	 * collection is not discarded.
    	 */
   	public final void handleMetadataToken(MetadataToken token) 
   		throws IllegalActionException {

         // ensure that each BToken arrives between the opening and closing
   	    	// delimiters of its collection  
	 	assert token.collection() == _collectionManager.collection():
		    "Token outside of collection: " +
		    "\n\t token = " + token + 
		    "\n\t token's collection = " + token.collection() +
		 	"\n\t current collection = " + _collectionManager.collection() + 	    
		    "\n\t actor = " + this;

	 	// apply parameter this actor if token is a ParameterToken
	 	if (token instanceof VariableToken) {
			_handleVariableToken((VariableToken) token);
	 	}
	 	
	   	// store and forward the token if the collection is ignored or the 
	 	// metadata handler so requests 
		TokenDisposition metadataDisposition = FORWARD_TOKEN;
		
	 	if (! _collectionManager.ignored() && 
 	        _collectionPath.exposeNonDataEvents()) {
 		    
	 	    try {
                metadataDisposition = 
                    _actor._handleMetadata(_collectionManager, token);
            } catch (CollectionException exception) {
		       	processCollectionException(exception);
	        }
	 	}
 		    
		if (metadataDisposition == FORWARD_TOKEN) {
	    		_collectionManager._storeInputMetadata(token);
	    }
	}	

	/** 
    	 * Handles opening delimiter tokens for collection actors. Creates a
    	 * new collection manager to handle each incoming collection signaled
    	 * by an opening delimiter and sets the disposition of the collection
    	 * according to whether the collection falls between the highest and 
    	 * lowest collection types to be processed and according to the return
    	 * value of handleCollectionStart if it does fall in that range. The 
    	 * new collection is discarded automatically if the parent collection
    	 * was discarded.
    	 *  
    	 * @param token The opening delimiter token.
    	 * @return DO_NOT_SEND_TOKEN in all cases. The collection manager will
    	 * send the token along with the rest of the collection later if the 
    	 * collection is not discarded.
    	 */
    public void handleOpeningDelimiter(OpeningDelimiterToken token) 
    		throws IllegalActionException {

        	// note that we are no longer above the lower collection types to 
        // ignore if an opening delimiter 	is received when the current 
        // collection type is the lowest collection type to process
	 	if (_aboveLowerIgnoredTypes && _collectionManager != null && 
 	        _collectionPath.ignoreCollectionsInside() != Object.class &&
 	        _collectionPath.ignoreCollectionsInside().
 	        		isInstance(_collectionManager.collection())) {	
   	    		
	 	    // indicate that we are no longer above the lower collection 
	 	    // types to ignore
	 	    _aboveLowerIgnoredTypes = false;
			
			// keep a reference to the collection above the the ignored 
	 	    // collections so that its closing delimiter can be recognized 
	 	    // later
			_lowestProcessedCollection = _collectionManager.collection();
		}
		
   		// push the current collection manager onto the collection
	 	// manager stack
   		_collectionManagerStack.push(_collectionManager);

   		// make the current collection manager the parent collection manager
   		_parentCollectionManager = _collectionManager;
   		
   		// create a new collection manager to manage the incoming collection
   		// represented by the delimiter
    		_collectionManager =  new IncomingCollectionManager(
	            token.collection(), _parentCollectionManager, 
    		            	_actor._getDefaultOutputPort());

    		// note that we are below the upper collection types to ignore if 
    		// the type of the incoming collection matches the highest 
    		// collection type to process
		if (! _belowUpperIgnoredTypes && 
	        _collectionPath.ignoreCollectionsOutside().
	        		isInstance(_collectionManager.collection())) {
			
		    	// set the flag indicating we are now below the highest 
		    	// collection type for processing
		    	_belowUpperIgnoredTypes = true;
			
		    	// keep a reference to this collection so that its closing 
		    	// delimiter can be recognized later
		    	_highestProcessedCollection = _collectionManager.collection();
		}
    		
		// discard the collection if the parent collection has been discarded
		if (_parentCollectionManager != null && 
		        _parentCollectionManager.discarded()) {
		    
		    	_collectionManager._discard();
		}
		
		// start processing the collection if it is below the upper ignored 
		// types and the lower ignored types
		if (_belowUpperIgnoredTypes && _aboveLowerIgnoredTypes) {

		    // call the collection start handler
		    CollectionDisposition disposition = PROCESS_AND_FORWARD_COLLECTION;

		    if (_collectionPath.exposeNonDataEvents()) {
	            try {
	                disposition = _actor._handleCollectionStart(_collectionManager);
	            } catch (CollectionException exception) {
			       	processCollectionException(exception);
					disposition = IGNORE_AND_FORWARD_COLLECTION;
	            }
		    }
		    
            // ignore the collection if so requested
		    if (disposition == IGNORE_AND_DISCARD_COLLECTION || 
		            disposition == IGNORE_AND_FORWARD_COLLECTION) {
		        
		        _collectionManager._ignore();
		    }
		    
		    // discard the collection if so requested
		    if (disposition == IGNORE_AND_DISCARD_COLLECTION || 
		            disposition == PROCESS_AND_DISCARD_COLLECTION) {
		        _collectionManager._discard();
		    }
		
		// otherwise ignore the collection
    		} else {
	    	    _collectionManager._ignore();
    		}
	}
         
    public void handleLoopTerminationToken(LoopTerminationToken
        token) throws IllegalActionException {
        
        _actor.handleLoopTermination(token);
    }   

    ///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
	
   	private final void _handleVariableToken(VariableToken variableToken) 
		throws IllegalActionException {
   	    	 	    
 	    String parameterName = variableToken.key();
 	    Parameter parameter = 
 	        (Parameter)_overridableParametersMap.get(parameterName);

 	    if (parameter != null) {
 	   		
 	        Token previousValue;
 	        if (_overriddenParametersMap.containsKey(parameter)) {
 	            previousValue = (Token)_overriddenParametersMap.get(parameter);
 	        } else {
	 	        // previous value is the token stored in the parameter if 
 	            // this is the first override of the parameter
	 	        previousValue = parameter.getToken();
 	        }
 	        
 	        Token newValue = (Token)variableToken.value();
 	        _collectionManager._overrideParameter(parameter, previousValue);
 	        _overriddenParametersMap.put(parameter, newValue);
 	        _actor._handleParameterChange(parameter, newValue);
 	    }    
   	}

    private void processCollectionException(CollectionException exception) {

        ExceptionDisposition exceptionDisposition = 
            _actor.handleCollectionException(_collectionManager, exception);
        
        if (exceptionDisposition == PROPAGATE_EXCEPTION) {
            _collectionManager.addException(_actor, exception);
            ((IncomingCollectionManager)_collectionManager)._ignore();
        }
    }
    
    public boolean managesTopOfScope(CollectionManager manager) {
        return manager.collection() == _highestProcessedCollection;
    }

   	private boolean _typeMatchesTarget(Token token) {
   	    
   	    if (token instanceof DataToken) {
   	        
            _unwrappedDataToken = ((DataToken)token).getValue();
            
	   	    return _collectionPath.targetDataType().
	   	    		isInstance(_unwrappedDataToken);
   	    
   	    } else if (token instanceof DomainObjectToken) {
   	        
   	        _unwrappedDomainObject = ((DomainObjectToken)token).object();
   	        
			return _collectionPath.targetDataType().
				isInstance(_unwrappedDomainObject);
			
   	    } else {
   	        
   	        _unwrappedDataToken = token;
   	        
	   	    return _collectionPath.targetDataType().
	   	    		isInstance(_unwrappedDataToken);
   	    }
   	}
   	
   	TestProbe createTestProbe() {
   	    return new TestProbe();
   	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	

    private final Coactor _actor;
    
    private boolean _aboveLowerIgnoredTypes = true;
    
    	private boolean _belowUpperIgnoredTypes;
    	
    	private IncomingCollectionManager _collectionManager = null;
    	
    private final CollectionPath _collectionPath;
        
    private final Stack _collectionManagerStack = new Stack();

	private Collection _highestProcessedCollection = null;
    	
   	private Collection _lowestProcessedCollection = null;
    
    private final Map _overriddenParametersMap;
    
    private final Map _overridableParametersMap;
        
	private IncomingCollectionManager _parentCollectionManager = null;

	private Token _unwrappedDataToken;

	private DomainObject _unwrappedDomainObject;
	
	///////////////////////////////////////////////////////////////////
	////                       public inner classes                ////
   	
	public static class CollectionDisposition {
	    	private CollectionDisposition() {}
	}
	
	public static class ExceptionDisposition {
	    	private ExceptionDisposition() {}
	}
	
    	public static class TokenDisposition {
		private TokenDisposition() {}
    }
    	
    	class TestProbe {
	
    	    public CollectionManager collectionManager() {
    	        return _collectionManager;
    	    }
            
        public Map overridableParametersMap() {
            return _overridableParametersMap;
        }
    	}
}
