/* Static methods for managing collection types.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp;

import java.util.HashMap;
import java.util.Map;

import org.nddp.phylogeny.NexusCollection;
import org.nddp.proteins.PDBCollection;
import org.nddp.proteins.PDBFooterCollection;
import org.nddp.proteins.PDBHeaderCollection;
import org.nddp.proteins.ProteinChainCollection;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.util.IllegalActionException;

////////////////////////////////////////////////////////////////////////// 
//// CollectionTypes 

/** 
 @author Timothy M. McPhillips
 */

public class CollectionTypes {

    private CollectionTypes() {}

	///////////////////////////////////////////////////////////////////
	////                       public methods                      //// 

    protected final static void register(String typeName, Class typeClass) {
        
	    	assert typeName != null: "Null typeName not allowed.";
	    	assert typeClass != null: "Null typeClass not allowed.";
	    	assert typeClass != NoMatch.class: "Class NoMatch not allowed.";
	    	assert typeClass == Object.class || 
			Collection.class.isAssignableFrom(typeClass): 
	    	    typeClass + " must be a subclass of Collection.";
	    	
	    	typeMap.put(typeName, typeClass);
    }
    
 	public final static boolean isDefined(String typeName) {
 	    
	    // explicitly disallow null typeName
	    if (typeName == null) {
		    throw new NullPointerException(
            		"Null typeName not allowed.");
		}
	    
	    // check for empty string specifically because it is defined in the
	    // _typeMap table but has a special meaning
	    if (typeName.equals("")) {
		    return false;
		}
	    
	    return typeMap.containsKey(typeName);
 	}
 	

 	public final static Class valueOfParameter(Parameter 
        collectionTypeParameter) throws IllegalActionException {

		if (collectionTypeParameter == null) {
		    throw new NullPointerException(
            		"Null collectionTypeParameter not allowed.");
		}

		return valueOfToken(collectionTypeParameter.getToken());
	}

	public final static Class valueOfString(String typeName) 
		throws IllegalActionException  {
	    
	    if (typeName == null) {
	    		throw new NullPointerException(
    		        "Null typeName not allowed.");
		}
	    
		Class typeClass = (Class)typeMap.get(typeName);

		if (typeClass == null) {
	    		throw new IllegalActionException(
    		        "No collection type " + typeName);
		}
		
		return typeClass;
    }

	public static Class valueOfToken(Token collectionTypeToken) 
		throws IllegalActionException {
	    
		if (collectionTypeToken == null) {
    		    throw new NullPointerException(
			    "Null collectionTypeToken not allowed.");
		}
		
		Class collectionType = null;
		
		try {
			collectionType = 
			    valueOfString(((StringToken)collectionTypeToken).stringValue());
		}
		catch(ClassCastException expected) {
		    	throw new IllegalActionException("collectionTypeToken must " +
	    	        "be a StringToken");
		}
		
		return collectionType;
	}

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 
	
    protected static void clear() {
	    typeMap.clear();
	}
    
    protected static void registerDefaultCollectionTypes() {

	    // map empty type name to Object class
    	    register("", Object.class);
    	    
    	    // add basic collection types
		register("Project", ProjectCollection.class);
		register("TextFile", TextFileCollection.class);
		
		// add domain-specific collection types
	    register("Nexus", NexusCollection.class);
    	    register("ProteinChain", ProteinChainCollection.class);
    	    register("PDB", PDBCollection.class);
    	    register("PDBHeader", PDBHeaderCollection.class);
    	    register("PDBFooter", PDBFooterCollection.class);
	}

    ///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private final static Map typeMap = new HashMap(); 	
	
	///////////////////////////////////////////////////////////////////
	////                    public inner classes                   ////

	public final static class NoMatch {}

	///////////////////////////////////////////////////////////////////
	////                   static initializers                     ////

	static { 
	    
	    registerDefaultCollectionTypes();
	}
}
