/* TODO One line description of class.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylip;

import java.util.List;
import java.util.Map;

import org.nddp.actors.EnvironmentActor;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.exceptions.ParseException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.phylip.PhylipFiles;
import org.nddp.util.Parameters;
import org.nddp.util.Port;
import org.nddp.util.ProcessEnvironment;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ArrayToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class ReadPhylipTrees extends EnvironmentActor {

    public ReadPhylipTrees(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an input port for receiving a map of taxon abbreviations
        taxonAbbreviations = new TypedIOPort(this, "taxonAbbreviations", true, false);
        taxonAbbreviations.setTypeEquals(BaseType.OBJECT);

        // create an output port for sending trees
        trees = new TypedIOPort(this, "tree", false, true);
        trees.setTypeEquals(new ArrayType(BaseType.OBJECT));

        // create a parameter for optionally specifying the name of the trees file
        fileName = 
            Parameters.stringParameter(this, "fileName", "outtree");

        // create a parameter for optionally specifying the name of the trees file
        treeNamePrefix = 
            Parameters.stringParameter(this, "treeNamePrefix", "Tree");

        // create a parameter for optionally specifying the root of the consensus tree
        rootAtNode = 
            Parameters.quotedStringParameter(this, "rootAtNode", "");
}

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort trees;
    public TypedIOPort taxonAbbreviations;

    public Parameter fileName;
    public Parameter treeNamePrefix;
    public Parameter rootAtNode;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    public final void attributeChanged(Attribute attribute) 
        throws IllegalActionException {
            
        if (attribute == fileName) {
            
            _fileName = Parameters.stringValue(fileName);
        
        } else if (attribute == treeNamePrefix) {
                
            _treeNamePrefix = Parameters.stringValue(treeNamePrefix);
             
        } else if (attribute == rootAtNode ) {
        
            _rootAtNode = Parameters.stringValue(rootAtNode);
        
        } else {
            super.attributeChanged(attribute);
        }
    }    
    
    public void fire() throws IllegalActionException {
 
        // call superclass method
        super.fire();
        
        // get a reference to the process environment
        ProcessEnvironment processEnvironment = receiveEnvironment();

        // get the optional abbreviations map
        Map abbreviationsMap = (Map)Port.getObject(taxonAbbreviations, 0, false);
        
        try {
            // read the output tree file
            String outtree = processEnvironment.readFile(_fileName);

            // parse the tree file and return a list of trees
            List treeList = PhylipFiles.parseTreeFileString(outtree,
                abbreviationsMap);

            // create an array for the trees
            Token[] tokenArray = new Token[treeList.size()];
            
            // iterate over trees in the list
            for (int i = 0; i < tokenArray.length; i++) {
                
                // get the next tree from the list
                Tree tree = (Tree)treeList.get(i);
                
                // optionally reroot the tree
                if (! _rootAtNode.equals("")) {
                    tree.reroot(_rootAtNode);
                }
                
                // set the name of the tree
                tree.setName(_treeNamePrefix);
                
                // store the tree in an object token and place in the token array
                tokenArray[i] = new ObjectToken(tree);
            }
            
            ArrayToken arrayToken = new ArrayToken(tokenArray);
            
            // send the arrayToken containing the trees
            trees.broadcast(arrayToken);
            
        } catch (ExternalEnvironmentException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (InconsistentDataException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (ParseException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        
        // forward the process environment
        sendEnvironment();
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                    private variables                      ////
    
    private String _fileName;
    private String _rootAtNode;
    private String _treeNamePrefix;
    private static final long serialVersionUID = 1L;
}
