/* Actor that creates an empty project collection from scratch.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors;

import org.nddp.Collection;
import org.nddp.CollectionManager;
import org.nddp.ProjectCollection;
import org.nddp.util.Parameters;

import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;

/**
 * An actor for spawning a top-level Project collection that may be used to
 * contain other collections.  It is fired only once during the execution 
 * of a workflow containing it.
 * 
 *  @author Timothy M. McPhillips
 * 
 * @nddp.coactor type="general"
 */
public class Project extends CollectionSource {
 
     /** 
     * Constructs an instance of Project.  The constructor initializes the 
     * project name parameter.
     * 
     * @param container The run-time container of this actor.
     * @param name The name of this actor.
     */  
    public Project(CompositeEntity container, String name)
        throws IllegalActionException, NameDuplicationException {
        
        // call the superclass constructor
        super(container, name);

        // create the project name parameter and set its default value to the 
        // null string
        projectName = Parameters.stringParameter(this, "projectName", "");
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    	//  The name of the project collection to be created by the actor.
    public Parameter projectName;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
     /** 
     * The method called by the director when the actor is fired. The method 
     * creates and sends a new top-level collection.  The collection is 
     * named according to the value of the projectName parameter which may 
     * be set by the user.
     * 
     * @throws IllegalActionException
     * @throws CollectionManagementException
     */  
    	public void fire() throws IllegalActionException {

        // call the superclass fire method
        super.fire();
        
        // extract the name of the project from the parameter
        String name = Parameters.stringValue(projectName);
        
        // create a new project top-level collection with the specified name
        Collection collection = new ProjectCollection(name);
        
        // create a collection manager for the new collection and associate
        // with the output port
        CollectionManager projectManager = 
            manageNewCollection(collection, null, output);
        
        // explicitly close the collection (since there is no parent collection to 
        // trigger automatic closure)
        projectManager.closeCollection();
    }
    
     /** 
     * The method called by the director after the actor has fired. This method
     * returns false to indicate that the actor should never be fired again, 
     * making this actor a one-shot.
     */  
    	public boolean postfire() throws IllegalActionException {
    
        // indicate to the director that the actor should never be fired again
        return false;
    }
}
