/* Actor that adds metadata tokens to a collection.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp.coactors;

import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.util.Parameters;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.Variable;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="general"
 */
public class SetMetadata extends CollectionTransformer {
	
	public SetMetadata(CompositeEntity container, String name)
		throws NameDuplicationException, IllegalActionException  {

		super(container, name);
		
        	metadataName = Parameters.stringParameter(this, "name", "");
        	metadataType = Parameters.stringParameter(this, "type", "STRING");
		priorValue = Parameters.stringParameter(this, "priorValue", "");
		posteriorValue = 
		    Parameters.stringParameter(this, "posteriorValue", "");
		
		_intValue = new Variable(this, "_intValue");
		_intValue.setTypeEquals(BaseType.INT);

		_doubleValue = new Variable(this, "_doubleValue");
		_doubleValue.setTypeEquals(BaseType.DOUBLE);
		
		_booleanValue = new Variable(this, "_booleanValue");
		_booleanValue.setTypeEquals(BaseType.BOOLEAN);
		
		_stringValue = new Variable(this, "_stringValue");
		_stringValue.setTypeEquals(BaseType.STRING);
	}
  
	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

	public Parameter metadataName;
	public Parameter metadataType;
	public Parameter posteriorValue;
	public Parameter priorValue;
	
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
 	
   	protected void _addMetadata(CollectionManager collectionManager, 
        String name, Token value) throws IllegalActionException {
    		
   	    collectionManager.addMetadata(name, value);
   	}
   	
    public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException {

        	_processValue(collectionManager, _posteriorValue);
   	}

    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException {

        	_processValue(collectionManager, _priorValue);
  	    
   	    	return CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
  	} 

    public void _handleParameterChange(Parameter parameter, Token newValue) 
		throws IllegalActionException {
            
   		if (parameter == metadataName) {
    			_variableName = Parameters.stringValue(newValue);
        	} else if (parameter == metadataType) {
 
        		String typeString = Parameters.stringValue(newValue);
        		
        		if (typeString.equals("INT")) {
        		    _value = _intValue;
        		} else if (typeString.equals("DOUBLE")) {
        		    _value = _doubleValue;
        		} else if (typeString.equals("BOOLEAN")) {
        			_value = _booleanValue;
        		} else if (typeString.equals("STRING")) {
        			_value = _stringValue;
        		} else {
        		    throw new IllegalActionException(
    		            "Variable type must be INT, DOUBLE, BOOLEAN," +
    		            " or STRING");
        		}
        	
        	} else if (parameter == priorValue) {
        		_priorValue = Parameters.stringValue(newValue);
        	} else if (parameter == posteriorValue) {
    			_posteriorValue = Parameters.stringValue(newValue);
        	} else {
            	super._handleParameterChange(parameter, newValue);
        	}
    }
   	
	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////

   	private void _processValue(CollectionManager collectionManager, 
        String metadataValueString) throws IllegalActionException {
   	    
   	    if (! metadataValueString.equals("")) {
   	    
	    		String expandedValue = 
	    		    collectionManager.expandMetadataSymbols(
	    		            metadataValueString);

	    		if (_value == _stringValue) {
  	            	_addMetadata(collectionManager, _variableName, 
  	            	        new StringToken(expandedValue));  	                    
  	        	} else {
	   	    		_value.setExpression(expandedValue);
	   	    		_addMetadata(collectionManager, _variableName, 
	   	    		        _value.getToken());
  	    		}
   	    }
   	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private Variable _booleanValue;
	private Variable _doubleValue;
	private Variable _intValue;
	private String _posteriorValue;
	private String _priorValue;
	private Variable _stringValue;
	private Variable _value;
	private String _variableName;
}
