/* Wrapper actor for Phylip CONSENSE.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
package org.nddp.coactors.phylogeny;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JFrame;
import javax.swing.JPanel;

import org.apache.batik.swing.JSVGCanvas;
import org.apache.batik.swing.gvt.GVTTreeRendererAdapter;
import org.apache.batik.swing.gvt.GVTTreeRendererEvent;

import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.DomainObject;
import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.phylogeny.phylip.PhylipDrawgramRunner;
import org.nddp.phylogeny.Tree;
import org.nddp.util.Parameters;
import org.nddp.util.ProcessEnvironment;
import org.nddp.util.ProcessRunner;
import org.nddp.util.ProcessRunningActor;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="phylogenetics"
 */
public class PhylipDrawgram extends ProcessRunningActor {

    public PhylipDrawgram(CompositeEntity container, String name)
        	throws NameDuplicationException, IllegalActionException {
        
        	super(container, name);
        
		windowTitle = 
		    Parameters.stringParameter(this, "windowTitle", "Drawgram");

		rootAtNode = 
		    Parameters.stringParameter(this, "rootAtNode", "");
		
        	Parameters.fix(collectionPath, "Nexus/Tree");
    }
    
	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    public Parameter rootAtNode;
    public Parameter windowTitle;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public void initialize() throws IllegalActionException {
    
        super.initialize();

        // start up the graphical user interface if it is not running
        if (gui.isRunning() == false) {
            gui.start();
        }
    }

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    public CollectionHandler.TokenDisposition _handleDomainObject(
        	CollectionManager fileCollectionManager, DomainObject object)
		throws IllegalActionException, CollectionException {

	    _tree = (Tree)object;
		
	    _runProgram();
		
		ProcessEnvironment environment = (ProcessEnvironment)_runResult;
	
		String plotfileURI =	"file:" + environment.workingDirectoryPath() + "/plotfile";
		
        	gui.loadCladogram(plotfileURI);

        environment.deleteWorkingDirectory();
        
        return  CollectionHandler.FORWARD_TOKEN;
    } 

    public void _handleParameterChange(Parameter parameter, Token newValue) 
		throws IllegalActionException {
            
        if (parameter == windowTitle) {
        		_windowTitle = Parameters.stringValue(newValue);
        } else if (parameter == rootAtNode) {
            _rootAtNode = Parameters.stringValue(newValue);
        } else {
        		super._handleParameterChange(parameter, newValue);
        }
    }
  
	///////////////////////////////////////////////////////////////////
	////                       protected methods                   ////

    protected void _configureRunner() throws InconsistentDataException {
	    PhylipDrawgramRunner runner = (PhylipDrawgramRunner)_runner;
	    runner.setTree(_tree);
	    runner.rootAtNode(_rootAtNode);
    }

    protected ProcessRunner _getRunnerInstance() {
        
        	return new PhylipDrawgramRunner();
    }

    private Gui gui = new Gui();
    private String _rootAtNode;
    private static final long serialVersionUID = 1L;
    private Tree _tree = null;
    private String _windowTitle;
	
	public class Gui {
	
	    public Gui() {
	    }
	    	
	    public boolean isRunning() {
		    return _isRunning;
		}
		
 		private boolean _isRunning = false;    
	   	private JFrame frame;
	    	private JSVGCanvas svgCanvas;
	
	    	public void start() {
	        
	        	frame = new JFrame(_windowTitle);
	    		svgCanvas = new JSVGCanvas();
	    	    JPanel panel = new JPanel(new BorderLayout());	
	        JPanel p = new JPanel(new FlowLayout(FlowLayout.LEFT));
	        panel.add("Center", svgCanvas);
	        	frame.getContentPane().add(panel);
	
	        	frame.addWindowListener(new WindowAdapter() {
	            public void windowClosing(WindowEvent e) {
	                _isRunning = false;
	            }
	        });

	        frame.setSize(400, 400);
	        frame.setVisible(true);
	        
            svgCanvas.addGVTTreeRendererListener(new GVTTreeRendererAdapter() {
	            public void gvtRenderingPrepare(GVTTreeRendererEvent e) {
	            }
	            public void gvtRenderingCompleted(GVTTreeRendererEvent e) {
	                synchronized(svgCanvas) {
	                    svgCanvas.notify();
	                }
	            }
	        });

            _isRunning = true;
	    	}
	    	
	    	public void loadCladogram(String path) {

	    	    svgCanvas.setURI(path);
			
			try {
		    	    synchronized(svgCanvas) {
					svgCanvas.wait();
				}
	        } catch (InterruptedException e) {
				// discard exceptions
	        }
	    	}
	}

}
