/* Wrapper actor for Phylip PARS.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors.phylogeny;

import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.DomainObject;
import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.DistanceMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.phylip.PhylipNeighborRunner;
import org.nddp.util.Parameters;
import org.nddp.util.ProcessRunner;
import org.nddp.util.ProcessRunningActor;

import ptolemy.data.IntToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="phylogenetics"
 */
public class PhylipNJ extends ProcessRunningActor {


    public PhylipNJ(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException {
        
        	super(container, name);

        outgroupIndex = Parameters.intParameter(this, "outgroupIndex", 1);
            
        Parameters.fix(collectionPath,"Nexus/");

        _enableParameterOverride(outgroupIndex);
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public Parameter outgroupIndex;

    ///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

    protected ProcessRunner _getRunnerInstance() {
        
        	return new PhylipNeighborRunner();
    }

    protected void _configureRunner() throws InconsistentDataException {

        PhylipNeighborRunner runner = (PhylipNeighborRunner)_runner;
        
        runner.useUPGMA(_useUPGMA);
        runner.setCharacterMatrix(_characterMatrix);
        runner.setDistanceMatrix(_distanceMatrix);
        runner.setOutgroup(_outgroupIndex);
    }
    
    public void _handleCollectionEnd(CollectionManager collectionManager) 
        throws IllegalActionException, CollectionException {
    
        if (collectionManager == _nexusCollectionManager) {
            
            if (_characterMatrix != null && _distanceMatrix != null) {
                _runProgram();
                _addTreeToCollection((Tree)_runResult);
            }
        }
    }

    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager
        collectionManager) throws IllegalActionException {
    
        _nexusCollectionManager = collectionManager;
        _characterMatrix = null;
        _treeCount = 0;

        return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
    }
    
    public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
            fileCollectionManager, DomainObject object) 
        throws CollectionException {

        if (object instanceof CharacterMatrix) {
            _characterMatrix = (CharacterMatrix)object;
        } else if (object instanceof DistanceMatrix) {
            _distanceMatrix = (DistanceMatrix)object;
        } else if (object instanceof Tree) {
            _treeCount++;
        }

        return  CollectionHandler.FORWARD_TOKEN;
    }
    
    public void _handleParameterChange(Parameter parameter, Token newValue) 
        throws IllegalActionException {
            
        if (parameter == outgroupIndex) {
                _outgroupIndex = Parameters.intValue(newValue);
        } else {
            super._handleParameterChange(parameter, newValue);
        }
    }   
    
    protected void useUPGMA(boolean useUPGMA) {
        _useUPGMA = useUPGMA;
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private methods                   ////
    
    private void _addTreeToCollection(Tree tree) {
        
        _nexusCollectionManager.addDomainObject(tree);

        _treeCount ++;
        
        _nexusCollectionManager.addMetadata(
            "treeCount", new IntToken(_treeCount));
    }
    
    private CharacterMatrix _characterMatrix;
    private DistanceMatrix _distanceMatrix;
    private CollectionManager _nexusCollectionManager;
    private int _outgroupIndex;
    private int _treeCount;
    private static final long serialVersionUID = 1L;
    private boolean _useUPGMA = false;
}
