/* Class representing a node in a phylogenetic tree.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.TreeSet;

import org.nddp.exceptions.ParseException;

public class TreeNode {

    private TreeNode(Tree tree, TreeNode parent) {
        
	    super();
	    
	    _tree = tree;
	    _parent = parent;
		_tree._nodeCount++;
	}
	
    // TODO This method is buggy and should be fixed so that 
    // Tree.clone() can be optimized.
	public TreeNode(Tree tree, TreeNode node, TreeNode parent) {
	    
	    this(tree, parent);
	    
	    _name = node.name();
	    _branchLength = node._branchLength;
        
        // TODO need to clone the character states
	    _characterStates = node._characterStates;
    
	    for (Iterator i = node._children.iterator(); i.hasNext(); ) {
	        _children.add(new TreeNode(_tree, (TreeNode)i.next(), this));
	    }
	}

    // TODO Add support for nameless terminal nodes
    private TreeNode(Tree tree, String newickNodeString, 
        TreeNode parentNode) throws ParseException {
        
        this(tree, parentNode);
        
        if (_tree._distancesDefined) {
            
                int colonIndex = newickNodeString.indexOf(':', 0);
                
                if (colonIndex == -1) {
                        throw new ParseException("Distance missing in node: " + 
                            newickNodeString);
                    } else {
                _setName(newickNodeString.substring(0,colonIndex));
                _branchLength = Double.parseDouble(
                    newickNodeString.substring(colonIndex + 1));
                    }
                
        } else {
            
            _setName(newickNodeString);
        }
    }
    
    public TreeNode(Tree tree, String name, TreeNode parent, 
            double branchLength) {
            
            this(tree, parent);
            
            _name = name;
            _branchLength = branchLength;
        }
        

    // TODO Add support for named internal nodes
    TreeNode(Tree tree, TreeNode parentNode, 
        double branchLength, String newickNodeString) throws ParseException {
 
        this(tree, null, parentNode, branchLength);

        for (int index = 0, childEndIndex; index < newickNodeString.length(); 
                index = childEndIndex + 1) {      
            
            if (newickNodeString.charAt(index) == '(') {
                
                int matchingParenthesisIndex = 
                    _findMatchingParenthesis(newickNodeString, index+1);
                
                childEndIndex = _findChildEnd(newickNodeString, 
                        matchingParenthesisIndex-1);
                
                double childBranchLength = 0;
                
                if (_tree._distancesDefined) {
                    try {
                        childBranchLength = Double.parseDouble(
                                newickNodeString.substring(
                                    matchingParenthesisIndex+1, 
                                    childEndIndex));
                        
                    } catch (StringIndexOutOfBoundsException e) {
                        throw new ParseException("Missing distance for node: "
                            + newickNodeString);
                    }
                }
                
                _children.add(new TreeNode(_tree,this, childBranchLength, 
                     newickNodeString.substring(index+1, matchingParenthesisIndex-1)));                  

            } else {
                    childEndIndex = _findChildEnd(newickNodeString, index);
            
                    _children.add(new TreeNode(tree,
                        newickNodeString.substring(index, childEndIndex), 
                                this));
            }
        }
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
	public double branchLength() { 
	    return _branchLength; 
    }

	public CharacterVector characterStates() { 
	    return _characterStates; 
	}

	public int equalityhashCode() {
	    
	    int result = 17;
	    
	    if (isTerminalNode()) {
	        result = 37 * result + _name.hashCode();
	    } else {
	        for (Iterator i = _children.iterator(); i.hasNext(); ) {
	            TreeNode child = (TreeNode)i.next();
	            result = 37 * result + child.equivalenceHashCode();
	        }
	    }
	    
	    return result;
	}		
	
	public int equivalenceHashCode() {
	    
	    int result = 17;
	    
	    if (isTerminalNode()) {
	        result = 37 * result + _name.hashCode();
	    } else {
	        
	        TreeSet sortedChildHashCodes = new TreeSet();
	        
	        for (Iterator i = _children.iterator(); i.hasNext(); ) {
	            TreeNode child = (TreeNode)i.next();
	            sortedChildHashCodes.add(
                    new Integer(child.equivalenceHashCode()));
	        }
	        
	        for (Iterator i = sortedChildHashCodes.iterator(); i.hasNext(); ) {
	            int hashCode = ((Integer)i.next()).intValue();
	            result = 37 * result + hashCode;
	        }
	    }
	    
	    return result;
	}
	
	public boolean isTerminalNode() {
		return _children.size() == 0;
	}
	
	public String name() { 
	    return _name; 
    }
	
    	public String nameForNewickString(boolean abbreviated, CharacterMatrix matrix) {
        
        if (abbreviated && matrix != null && _name != null && ! _name.equals("")) {
        		return String.valueOf(matrix.taxonIndex(_name));
        	} else {
    			return _name;
        	}
    	}
    	
	public TreeNode parent() { 
	    return _parent; 
	}
	
	public void print(String spacer) {
        	System.out.println(spacer + "(" + _branchLength + ") " + _name + 
    	        " " + _characterStates);
        	spacer += "  ";
    		for (Iterator i = _children.iterator(); i.hasNext(); ) {
        		((TreeNode)i.next()).print(spacer);
        	}
	}
    
	public String toString() {
    	    return toString(false, false, null);
    	}
    	    
    	public String toString(boolean abbreviated, boolean hideLength,
            CharacterMatrix matrix) {
        
        	StringBuffer buffer = new StringBuffer();
        
        	if (isTerminalNode()) {

        	    buffer.append(nameForNewickString(abbreviated, matrix));
            
        	} else {

        	    boolean isFirst = true;
            	buffer.append("(");
            
            	if (this == _tree._root) {
                 String name = nameForNewickString(abbreviated, matrix);
                 if (name != null) {
                     buffer.append(name);
                    isFirst = false;
                 }
            	}
            
            	for (Iterator i = _children.iterator(); i.hasNext(); 
        			isFirst = false) {
            	    
                	if (! isFirst) buffer.append(",");
                	buffer.append(((TreeNode)i.next()).toString(abbreviated, 
            	        hideLength, matrix));
            	}
            	buffer.append(")");
        	}
        
        	if (! hideLength && _tree._distancesDefined && _parent != null) {
        	    buffer.append(":" + _branchLength);
        	}
        	
        	return buffer.toString();
	}
    ///////////////////////////////////////////////////////////////////
    ////                         protected methods                 ////
        
    protected void reroot(TreeNode newParent) {

        // save a reference to the current parent node 
        TreeNode oldParent = _parent;
        
        // assign the new parent for this node
        _parent = newParent;
    
        // find the new parent in the list of child nodes and
        // remove it from the list
        for (Iterator i = _children.iterator(); i.hasNext(); ) {
                TreeNode child = (TreeNode)(i.next());
                if (child == newParent) {
                    i.remove();
                    break;
            }
        }

        if (oldParent != null) {
            
            if (oldParent == _tree._oldRoot && 
                    oldParent._children.size() == 2) {

                for (Iterator i = 
                    _tree._oldRoot._children.iterator(); 
                    i.hasNext(); ) {
                    
                    TreeNode child = (TreeNode)(i.next());
                    if (child != this) {
                        _children.add(child);
                        child._parent = this;
                        _tree._oldRoot = null;
                        break;
                    }
                }
                
        } else {            
            // add the old parent as a child of this node
                _children.add(oldParent);
                
                // reroot the old parent with this node as it's parent
            oldParent.reroot(this);
            }
        }
    }

    protected void _setName(String name) {
        
        _name = _tree._expandName(name);
        
        _tree._nameToNodeMap.put(_name, this);
    }

	///////////////////////////////////////////////////////////////////
	////                         protected variables               ////
	
	protected double _branchLength = Double.NaN;
	protected CharacterVector _characterStates;
	protected final List _children = new LinkedList();
	protected String _name = "";
	protected TreeNode _parent;
	protected final Tree _tree;
    
    ///////////////////////////////////////////////////////////////////
    ////                         private methods                   ////

    private int _findChildEnd(String newickString, int index) {

        int nextCommaIndex = newickString.indexOf(',', index);
        
        if (nextCommaIndex == -1) {
            nextCommaIndex = newickString.length();
        }
        
        return nextCommaIndex;
    }

    private int _findMatchingParenthesis(String newickString, int index) {
        
        int leftParentheses = 1;
        int rightParentheses = 0;
        
        while (leftParentheses > rightParentheses) {
            
            assert index < newickString.length(): 
                "Matching parenthesis not found in " + newickString;
            
                char character = newickString.charAt(index);
                
                if (character == '(') {
                    leftParentheses++; 
                } else if (character == ')') {
                    rightParentheses++;
                }
                index++;
        }
        
        return index;
    }
}
