/* Program runner for Phylip PARS.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny.phylip;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.ExternalApplicationException;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.DistanceMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.util.ProcessEnvironment;
import org.nddp.util.ProcessRunner;

public class PhylipNeighborRunner extends ProcessRunner {

    public PhylipNeighborRunner() {
        
        super();
    }

    ///////////////////////////////////////////////////////////////////
	////                         public methods                    ////    	

    public Object run() throws CollectionException, ExternalEnvironmentException {
        
        ProcessEnvironment environment = _createProcessEnvironment("NEIGHBOR");
        
        _writeInputFiles(environment);
        
        _startProgram(environment, 
                System.getProperty("PHYLIP_EXE_DIR") + "/neighbor");
        
        _writeStandardInput();

        _waitForProgramCompletion();
        
        Tree tree = (Tree)_parseOutput(environment);
        
//        _destroyEnvironment(environment);
        
        return tree;
    }

    public void setCharacterMatrix(CharacterMatrix characterMatrix) {
        _characterMatrix = characterMatrix;
    }
	
    public void setDistanceMatrix(DistanceMatrix distanceMatrix) {
        _distanceMatrix = distanceMatrix;
    }

    public void setOutgroup(int outgroupIndex) {
        _outgroupIndex = outgroupIndex;
    }

    public void useUPGMA(boolean useUPGMA) {
        _useUPGMA = useUPGMA;
    }
    
    ///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

    	protected Tree _parseOutput(ProcessEnvironment environment) 
    		throws CollectionException {
            
        // Read the output tree file
        String outtree = environment.readFile("outtree");

        // Parse the tree file
        List treeBufferList = PhylipFiles.parseTreeFileString(outtree,
            _taxonAbbreviations);
        
        return (Tree)treeBufferList.get(0);
        
    	}
    	
    protected void _writeInputFiles(ProcessEnvironment environment)
            throws CollectionException {
        
        try {
            _taxonAbbreviations = PhylipFiles.writeDistanceMatrixFile(
                _characterMatrix, 
                _distanceMatrix, 
                environment.createInputFile("infile"));
        } catch (IOException e) {
            throw new ExternalApplicationException(
                "Error writing to input file.");
        }
    }
    
    protected void _writeStandardInput() 
    		throws ExternalApplicationException {
        
		try {
            
            if (_useUPGMA) {
                _writeToProcess("N");
            }
            
	    		// input is a lower diagonal matrix
	    		_writeToProcess("L");		    		
	    
            // optionally specify taxon to use as outgroup
            if (_outgroupIndex > 0) {
                _writeToProcess("O");
                _writeToProcess(String.valueOf(_outgroupIndex));
            }
 
            // write out data at beginning of run
            _writeToProcess("1");

            // Accept configuration and begin execution
            _writeToProcess("Y");    
             
		} catch (IOException e) {
			throw new ExternalApplicationException(
		        "Error writing to process");
        	}		    
    }
    
    private DistanceMatrix _distanceMatrix;
    private CharacterMatrix _characterMatrix;
    private int _outgroupIndex; 
    private Map _taxonAbbreviations;
    private boolean _useUPGMA = false;
}

