/* Modified version of Ptolemy class for handling files and URLs.
 
Copyright (c) 2004 The Regents of the University of California.
All rights reserved.
Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

PT_COPYRIGHT_VERSION_2
COPYRIGHTENDKEY

*/

package org.nddp.util;

import ptolemy.kernel.util.*;
import ptolemy.util.StringUtilities;
import ptolemy.kernel.util.NamedObj;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;


public class FileOrUrl extends NamedObj {

    public FileOrUrl( String name ) {
   
        _name = name;
   
    }

    public BufferedReader open() throws IllegalActionException, NameDuplicationException {
    
        // check if file name refers to standard input
        if (_name.trim().equals("System.in")) {
        
            // create a buffered reader to the standard input stream
            _reader = new BufferedReader(new InputStreamReader(System.in));
            
        } else {

            // create a URL object from the file name
            URL url = createURL();
            
            //  throw an exception if the file name was empty
            if ( url == null ) {
                throw new IllegalActionException(this, "No file name has been specified.");
            }
            
            // create a BufferedReader object from the URL
            try {
                _reader = new BufferedReader(new InputStreamReader(url.openStream()));
            } catch (IOException ex) {
                throw new IllegalActionException(this, ex, "Cannot open file or URL");
            }        
        }
        
        // return the buffered reader
        return _reader;
    }


    /** Return the file as a URL.  If the file name is relative, then
     *  it is interpreted as being relative to the directory returned
     *  by getBaseDirectory(). If the name begins with "$CLASSPATH",
     *  then search for the file relative to the classpath.
     *  If no file is found, then it throws an exception.
     *  @return A URL, or null if no file name or URL has been specified.
     *  @exception IllegalActionException If the file cannot be read, or
     *   if the file cannot be represented as a URL (e.g. System.in).
     */
    public URL createURL() throws IllegalActionException {
    
        // replace special strings in file name
        _name = substituteSpecialStrings(_name);

        // return null if file name string is empty
        if (_name == null || _name.trim().equals("")) {
            return null;
        }

        // create a File object from the file name string to access File methods
        File file = new File(_name);
        
        // extract the unqualified file name
        _unqualifiedName = file.getName();
        
        // assume file is local (not a web URL) if it hass an absolute path
        if (file.isAbsolute()) {
        
            // throw an exception if file is not readable
            if (!file.canRead()) {
                throw new IllegalActionException(this, "Cannot read file: " + _name);
            }
            
            _length = file.length();
            
            // create a URL object from the file object
            try {
                return file.toURL();
            } catch (MalformedURLException ex) {
                throw new IllegalActionException(this, "Cannot open file: " + ex.toString());
            }
            
        // otherwise assume that the file is a reference to a web URL
        } else {

            try {
                return new URL(_name);
            } catch (MalformedURLException e) {
                throw new IllegalActionException(this, "Unable to open as a file or URL: " + _name);
            }
        }
    }

    
    public long length() {
        return _length;
    }
    
    /** Close the file. If it has not been opened using openForReading()
     *  or openForWriting(), then do nothing.  Also, if the file is
     *  System.in or System.out, then do not close it (it does not make
     *  sense to close these files).
     *  @exception IllegalActionException If the file or URL cannot be
     *   closed.
     */
    public void close() throws IllegalActionException {
        if (_reader != null) {
            if (_reader != _stdIn) {
                try {
                    _reader.close();
                } catch (IOException ex) {
                    // This is typically caused by the stream being
                    // already closed, so we ignore.
                }
            }
        }
    }


    public static String substituteSpecialStrings(String name) {
    	
        String result = name;
        
        if (result.indexOf("$PTII") >= 0) {
            result = StringUtilities.substitute(result,
                    "$PTII",
                    StringUtilities.getProperty("ptolemy.ptII.dir"));
        }
        
        if (result.indexOf("$HOME") >= 0) {
            result = StringUtilities.substitute(result,
                    "$HOME",
                    StringUtilities.getProperty("user.home"));
        }
        
        if (result.indexOf("$CWD") >= 0) {
            result = StringUtilities.substitute(result,
                    "$CWD",
                    StringUtilities.getProperty("user.dir"));
        }
        
        return result;
    }


    public String unqualifiedFileName() {
        return _unqualifiedName;
    }

    public String name() {
        return _name;
    }
    
    public static String fileRoot(String fileName) {
        
		int fileExtensionIndex = fileName.lastIndexOf('.');

		if (fileExtensionIndex != -1) {
        		fileName = fileName.substring(0,fileExtensionIndex);
        	}
		
		return fileName;
    }
		
    public void setName(String name) throws IllegalActionException, NameDuplicationException {} 

    private long _length = 0;
    private String _name;
    private String _unqualifiedName = null;
    private BufferedReader _reader;
    private static BufferedReader _stdIn = null;

}



