/* Convenient methods for creating and using actor partameters.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.nddp.CollectionTypes;

import ptolemy.data.ArrayToken;
import ptolemy.data.BooleanToken;
import ptolemy.data.DoubleToken;
import ptolemy.data.IntToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.expr.Variable;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.Settable;


/**
 * An uninstantiable class that provides static methods for creating and 
 * accessing user-configurable parameters for actors.  Factory methods
 * are provided for constructing and configuring parameters to hold 
 * different types of tokens.  Corresponding methods are provided
 * for extracting values from the parameters later.  Together, these methods
 * simplify the use of Ptolemy parameters by hiding their underlying 
 * complexities, especially those of the array token types.
 * 
 * @author Timothy M. McPhillips
 */

public class Parameters {

	/** Private constructor prevents instantiation. */
    private Parameters() {}
	
	
    /** 
     * Creates a boolean parameter.  The value can be extracted later 
     * using the method {@link #booleanValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * @param value The initial value of the parameter.
     * 
     * @return The new parameter.
     */
	public final static Parameter booleanParameter(NamedObj actor, 
        String label, boolean value) throws IllegalActionException, 
        NameDuplicationException {
	    
	    // construct and return the new parameter
	    return new Parameter(actor, label, new BooleanToken(value));
	}


	public final static boolean booleanValue(Token token) 
		throws IllegalActionException {
	    
	    // extract and return the boolean value stored in the parameter
	    return ((BooleanToken)token).booleanValue();
	}
	
	/** 
     * Extracts the value of the passed boolean Parameter.  
     * 
     * @param value The parameter to evaluate.
     * @return The boolean value of the passed Parameter.
     * @exception IllegalActionException If the parameter is not a boolean parameter.
     */	
	public final static boolean booleanValue(Variable parameter) 
		throws IllegalActionException {
	    
	    // extract and return the boolean value stored in the parameter
	    return booleanValue(parameter.getToken());
	}
	
	public static Set collectionTypeSetValue(Token token) 
		throws IllegalActionException {
        
	   	// create a set to contain the collection types
        	Set set = new HashSet();
        
        // extract the array token from the parameter
        	ArrayToken array = (ArrayToken)token;
    
        	// interate over the tokens in the array token
        	for (int i = 0; i < array.length(); i++ ) {
            
            	// extract the value of the string in the token
            	String value = ((StringToken)array.getElement(i)).stringValue();
            
            	// get a CollectionType object representing the type named by 
            	// the string
            	Class type = CollectionTypes.valueOfString(value);
            
            	// add the type to the set if it is not NULL_COLLECTION_TYPE
            	if (type != Object.class) {
                	set.add(type);
        		}
        	}
        
        	// return the set of collection types
        	return set;
    }
    
	/** 
     * Extracts the strings stored in the passed string array Parameter and 
     * returns a Set containing a CollectionType object for each string.  
     * Each string must name a known CollectionType. 
     * 
     * @param parameter The parameter to evaluate.
     * @return A Set containing the CollectionType objects named in the passed
     * Parameter.
     * @exception IllegalActionException If the parameter does not contain an array of
     * strings.
     */
	public static Set collectionTypeSetValue(Variable parameter) 
		throws IllegalActionException {
 
        	// return the set of collection types
        	return collectionTypeSetValue(parameter.getToken());
    }
 
    /** 
     * Creates an double parameter.  The value can be extracted later 
     * using the method {@link #doubleValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * @param value The initial value of the parameter.
     * 
     * @return The new parameter.
     */	
	public final static Parameter doubleParameter(NamedObj actor, 
        String label, double value) throws IllegalActionException, 
        NameDuplicationException {
	    
	    // construct and return the new parameter
	    return new Parameter(actor, label, new DoubleToken(value));
	}
	
	
	public final static double doubleValue(Token token) 
		throws IllegalActionException {
	    
	    // extract and return the double value stored in the parameter
	    return ((DoubleToken)token).doubleValue();
	}	
	
	/** 
     * Extracts the value of the passed double Parameter.  
     * 
     * @param parameter The parameter to evaluate.
     * @return The double value of the passed Parameter.
     * @exception IllegalActionException If the parameter is not an double parameter.
     */	
	public final static double doubleValue(Variable parameter) 
		throws IllegalActionException {
	    
	    // extract and return the double value stored in the parameter
	    return doubleValue(parameter.getToken());
	}	
	
	public static void fix(Parameter parameter, String value) {
	    parameter.setExpression(value);
	    parameter.setVisibility(Settable.NOT_EDITABLE);
	}

    /** 
     * Creates an integer parameter.  The value can be extracted later 
     * using the method {@link #intValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * @param value The initial value of the parameter.
     * 
     * @return The new parameter.
     */	
	public final static Parameter intParameter(NamedObj actor, 
        String label, int value) throws IllegalActionException, 
        NameDuplicationException {
	    
	    // construct and return the new parameter
	    return new Parameter(actor, label, new IntToken(value));
	}	
	
	public final static int intValue(Token token) 
		throws IllegalActionException {

	    // extract and return the integer value stored in the parameter
	    return ((IntToken)token).intValue();
	}	
              
	/** 
     * Extracts the value of the passed integer Parameter.  
     * 
     * @param parameter The parameter to evaluate.
     * @return The int value of the passed Parameter.
     * @exception IllegalActionException If the parameter is not an int parameter.
     */	
	public final static int intValue(Variable parameter) 
		throws IllegalActionException {

	    // extract and return the integer value stored in the parameter
	    return intValue(parameter.getToken());
	}

	/** 
     * Creates a string array parameter with a default value of one empty string.
     * The string array can be extracted later using the method 
     * {@link #stringArrayValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * 
     * @return The new parameter.
     */
	//TODO delete this method when above method has been updated
	public final static Parameter stringArrayParameter(NamedObj actor,
        String label) throws IllegalActionException, NameDuplicationException {
		
	    // construct the new parameter with an empty string as its initial value
		return stringArrayParameter(actor, label, "{\"\"}");
	}
	
	/** 
     * Creates a string array parameter.  The string array can be extracted
     * later using the method {@link #stringArrayValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * @param value The initial value of the string array.
     * 
     * @return The new parameter.
     */
	//TODO Make third parameter a String array
	public final static Parameter stringArrayParameter(NamedObj actor, 
        String label, String value) throws IllegalActionException, 
        NameDuplicationException {

	    // construct the new parameter
		Parameter parameter = new Parameter(actor, label);

		// initialize the parameter value
		parameter.setExpression(value);
		
		// specify that the parameter is a string token array
		parameter.setTypeEquals(new ArrayType(BaseType.STRING));
		
        	// return the new parameter
		return parameter;
	}
    	
	public static String[] stringArrayValue(Token token) 
    		throws IllegalActionException {
        
	    	// extract the array token from the parameter
        	ArrayToken array = (ArrayToken)token;
        	
        	// create a string array with length matching that of the array token
    		String[] stringArray = new String[array.length()];
    	
    		// iterate over the elements of the array token
        	for (int i = 0; i < array.length(); i++ ) {
            
            	// extract the string from the array token element and store in the array
            	stringArray[i] = ((StringToken)array.getElement(i)).stringValue();
        	}
        	
        	// return the array of strings
        	return stringArray;
    }
    
	/** 
     * Extracts the string elements of the passed string array Parameter.  The order of
     * the array elements is preserved.  
     * 
     * @param parameter The parameter to evaluate.
     * @return An array of strings representing the value of the passed Parameter.
     * @exception IllegalActionException If the parameter does not contain an array of
     * strings.
     */	
	public static String[] stringArrayValue(Variable parameter) 
		throws IllegalActionException {
	    
	    return stringArrayValue(parameter.getToken());
	}
    
    
    public final static Parameter stringEnumerationParameter(NamedObj actor, 
            String label, String[] allowedValues, String initialValue) 
            throws IllegalActionException, NameDuplicationException {
        
        // construct the new parameter
        Parameter parameter = new StringParameter(actor, label);

        // initialize the parameter value
        parameter.setExpression(initialValue);
        
            // return the new parameter
        return parameter;    
    }
    
	
    public static Map stringMapValue(Token token) 
    		throws IllegalActionException {
        
        // extract the array token from the parameter
        	ArrayToken array = (ArrayToken)token;
    
        	// make sure there is an even number of tokens in the 
        	// array token or exactly one
        	if ( array.length() % 2 != 0 && array.length() != 1)
        	    throw new IllegalActionException(
    	            "Array must contain an even number of tokens "
        	        + "or a single empty string.");
    
        	// if there is just one token, make sure it is an empty string
        	if (array.length() == 1 ) {
        	    
        	    // extract string value from the only token in the array token
        	    String onlyString = 
        	        ((StringToken)array.getElement(0)).stringValue();
        	    
        	    // make sure the string is the empty string
        	    if (! onlyString.equals(""))
	        	    throw new IllegalActionException(
    	            		"Array must contain an even number of tokens "
        	         	+ "or a single empty string.");
        	}
        	
    		// create a map to hold the key-value pairs to be extracted from
        	// the strings
        	Map map = new HashMap();
        
        	// iterate over pairs of tokens in the array token
        	for (int i = 0; i < array.length() - 1; i += 2) {
        	    
        	    // extract the key string from the first token in the pair
            	String key = ((StringToken)array.getElement(i)).stringValue();
            	
            	// extract the value string from the second token in the pair
            	String value = ((StringToken)array.getElement(i+1)).stringValue();
            	
            	// store the key and value pair in the map
	    		map.put(key,value);
        	}   
    
        	// return the map of key-value pairs
        	return map;
    }

	/** 
     * Extracts the individual string values of the passed string array 
     * Parameter and builds a map relating adjacent pairs of elements.  
     * The first element in each pair becomes the key the map, and the 
     * second element becomes the value the map.  A single empty string 
     * may be used to represent a null list of key-value pairs.
     * 
     * @param parameter The parameter to evaluate.
     * @return The Map built up from the pairs of strings in the array Parameter.
     * @exception IllegalActionException If the parameter does not contain an array of
     * strings or if the array has an odd number of elements.
     */	
	public static Map stringMapValue(Variable parameter) 
		throws IllegalActionException {
	    
	    return stringMapValue(parameter.getToken());
    }
    	
	/** 
     * Creates a string parameter and initializes it with the value passed.  
     * The value of the parameter can be extracted later 
     * using the method {@link #stringValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * @param value The initial value of the parameter.
     * 
     * @return The new parameter.
     */
	public final static Parameter stringParameter(NamedObj actor, 
        String label, String value) throws IllegalActionException, 
        NameDuplicationException {
	    
	    // construct and return the new parameter
	    Parameter parameter = new StringParameter(actor, label);
	    parameter.setExpression(value);
	    return parameter;
	}

    public final static Parameter quotedStringParameter(NamedObj actor, 
            String label, String value) throws IllegalActionException, 
            NameDuplicationException {
            
            // construct and return the new parameter
            Parameter parameter = new Parameter(actor, label);
            parameter.setTypeEquals(BaseType.STRING);
            parameter.setExpression("\"" + value + "\"");
            return parameter;
        }
        
	/** 
     * Extracts the strings stored in the passed string array Parameter 
     * and returns them in a Set.  
     * 
     * @param parameter The parameter to evaluate.
     * @return A Set containing the strings in the passed Parameter.
     * @exception IllegalActionException If the parameter does not contain 
     * an array of strings.
     */
    public static Set stringSetValue(Variable parameter) 
    		throws IllegalActionException {
        
        // create a set to contain the strings in the parameter
        Set set = new HashSet();
        
        // extract the array token from the parameter
        ArrayToken array = (ArrayToken)parameter.getToken();
    
        // iterate over the tokens in the array token 
        for (int i = 0; i < array.length(); i++ ) {
            
            	// extract the value of the string in the token
            String value = ((StringToken)array.getElement(i)).stringValue();
            
            // add the string to the set
            set.add(value);
        }
        
        // return the set of strings
        return set;
    }

	public final static String stringValue(Token token) 
		throws IllegalActionException {
	    
	    // extract and return the string value stored in the parameter
		return ((StringToken)token).stringValue();
	}
	
	/** 
     * Extracts the value of the passed string Parameter. 
     *  
     * @param parameter The parameter to evaluate.
     * @return The String value of the passed Parameter.
     * @exception IllegalActionException If the parameter is not a string parameter.
     */	
	public final static String stringValue(Variable parameter) 
		throws IllegalActionException {
	    
	    // extract and return the string value stored in the parameter
		return stringValue(parameter.getToken());
	}
	
	/** 
     * Creates a array parameter than can hold any type of value.  The array of tokens
     * can be extracted later uisng the method {@link #tokenArrayValue(Parameter)}.
     * 
     * @param actor The actor containing the parameter to be created.
     * @param label The parameter label of the parameter to be created.
     * @param value The initial value of the array.
     * 
     * @return The new parameter.
     */
	public final static Parameter tokenArrayParameter(NamedObj actor, 
        String label, String value) throws IllegalActionException, 
        NameDuplicationException {
	
	    // construct the new parameter
	    Parameter parameter = new Parameter(actor, label);
	    
	    // initialize the parameter value
        	parameter.setExpression(value);
        	
        	// specify that the parameter is a generic token array
        	parameter.setTypeEquals(new ArrayType(BaseType.UNKNOWN));
        
        	// return the new parameter
        	return parameter;
	}
   
    public static ArrayToken tokenArrayValue(Token token) 
    		throws IllegalActionException {
        
    	    // extract and return the array token from the parameter
    	    return (ArrayToken)token;
    	}

	/** 
     * Extracts the ArrayToken stored in the passed Parameter.  
     * 
     * @param parameter The Parameter to evaluate.
     * @return The ArrayToken extracted from <code>parameter</code>.
     * @exception IllegalActionException If the parameter does not 
     * contain an ArrayToken;
     */
    	public static ArrayToken tokenArrayValue(Variable parameter) 
    		throws IllegalActionException {
    	    
    	    return tokenArrayValue(parameter.getToken());
    	}
}
