/* An actor that uses the ImageJ system for displaying an image. This
   actor is based on the ImageReader actor.

   Dan Higgins - NCEAS

   @Copyright (c) 2001-2004 The Regents of the University of California.
   All rights reserved.

   Permission is hereby granted, without written agreement and without
   license or royalty fees, to use, copy, modify, and distribute this
   software and its documentation for any purpose, provided that the
   above copyright notice and the following two paragraphs appear in all
   copies of this software.

   IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
   FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
   ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
   THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
   INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
   MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
   PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
   CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
   ENHANCEMENTS, OR MODIFICATIONS.

   PT_COPYRIGHT_VERSION 2
   COPYRIGHTENDKEY
*/

package util;

import ij.ImageJ;
import ij.ImagePlus;

import java.io.File;
import java.net.URL;

import ptolemy.actor.lib.Sink;
import ptolemy.data.StringToken;
import ptolemy.data.expr.FileParameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.Workspace;

//////////////////////////////////////////////////////////////////////////
//// ImageJActor
/**
 * This actor launches the ImageJ package from NIH using the
 * input file. see http://rsb.info.nih.gov/ij/ for information
 * on the ImageJ system for image processing
 *
 * This actor simply extends Sink; it thus has an 'input' multiport
 * The input is assumed to a file name which ImageJ will try to open.
 * ImageJ will succeed in opening a number of different types of image files.
 * 
 * When this actor fires, the image file input should be displayed along with
 * a small 'ImageJ' window with a set of menu items. These menu can be used to
 * apply a number of image processing operation to the displayed image.
 *  @author  Dan Higgins
 */
public class ImageJActor extends Sink {

    /** Construct an actor with the given container and name.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public ImageJActor(CompositeEntity container, String name)
            throws IllegalActionException, NameDuplicationException {
        super(container, name);


        fileOrURL = new FileParameter(this, "fileOrURL");
    }


    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** The file name or URL from which to read.  This is a string with
     *  any form accepted by File Attribute.
     *
     *  The file name should refer to an image file that is one of the
     *  many image files that ImageJ can display. These include tiffs, gifs,
     *  jpegs, etc. See the ImageJ Help menu or http://rsb.info.nih.gov/ij/
     */
    public FileParameter fileOrURL;

    
    public static ImageJ ij;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** If the specified attribute is <i>URL</i>, then close
     *  the current file (if there is one) and open the new one.
     *  @param attribute The attribute that has changed.
     *  @exception IllegalActionException If the specified attribute
     *   is <i>URL</i> and the file cannot be opened.
     */
    public void attributeChanged(Attribute attribute)
            throws IllegalActionException {
        if (attribute == fileOrURL) {
            // Would it be worth checking to see if the URL exists and
            // is readable?
            _file = fileOrURL.asFile();
            if (_file!=null) {
              try{
                _file.createNewFile();  // creates a new empty file if one does not exist
              } catch (Exception e) {
                throw new IllegalActionException("problem creating output file (in attributeChanged method)");
              }
            }
            _url = fileOrURL.asURL();
        }
        super.attributeChanged(attribute);
    }

    /** Clone the actor into the specified workspace. This calls the
     *  base class and then set the filename public member.
     *  @param workspace The workspace for the new object.
     *  @return A new actor.
     *  @exception CloneNotSupportedException If a derived class contains
     *   an attribute that cannot be cloned.
     */
    public Object clone(Workspace workspace)
            throws CloneNotSupportedException {
        ImageJActor newObject = (ImageJActor)super.clone(workspace);
        //newObject.output.setMultiport(true);
        return newObject;
    }

    /** Open the file at the URL, and set the width of the output.
     */
    public void initialize() throws IllegalActionException {
        attributeChanged(fileOrURL);
    }

    /** Read in an image.
     *  @exception IllegalActionException If an IO error occurs.
     */
    public boolean prefire() throws IllegalActionException {
        if (_url == null) {
          _fileRoot = null;
        }
        else{
          try{
          _fileRoot = _url.getFile();
          } catch (Exception e) {
          _fileRoot = null;
          }
        }  
        return super.prefire();
    }

  /**
   *
   */
  public synchronized void fire() throws IllegalActionException {
    super.fire();
    if (!input.hasToken(0)) return;
    
    // If the fileOrURL input port is connected and has data, then
    // get the file name from there.
    if (input.getWidth() > 0) {
      if (input.hasToken(0)) {
        String name = ((StringToken)input.get(0)).stringValue();
        // Using setExpression() rather than setToken() allows
        // the string to refer to variables defined in the
        // scope of this actor.
        fileOrURL.setExpression(name);
        _url = fileOrURL.asURL();
        _fileRoot = _url.getFile();
      }
    }
    
    System.out.println("firing ImageJActor");
    
    if (ij == null) {
      if (IJMacro.ij!=null) {// IJMacro may already have a static instance of an ImageJ class; if so, use it
        ij = IJMacro.ij;
      } else {
        ij = new ImageJ();
      }
    }
    if (ij!=null && !ij.isShowing()) {
      ij.show();
    }
    if (_fileRoot!=null) {
      new ImagePlus(_fileRoot).show();
    }
  }

    ///////////////////////////////////////////////////////////////////
    ////                         private members                   ////

    // The URL as a string.
    private String _fileRoot;

    // The File
    private File _file;
    
    // The URL of the file.
    private URL _url;
}
