/**
 *  '$RCSfile: ZipFiles.java,v $'
 *  Copyright: 2000 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *    Authors: @authors@
 *    Release: @release@
 *
 *   '$Author: higgins $'
 *     '$Date: 2006/03/20 22:16:48 $'
 * '$Revision: 1.3 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;

import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import java.util.Date;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.ArrayToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import util.PersistentVector;

/**
 *         Name: ZipFiles.java
 *      Purpose: The purpose of this actor is to 'zip' multiple files into
 *               a single zipped archive
 *
 *      input is a string array of file names to be zipped
 *      and desired name of the output zip file
 *      Zip file name will be output when finished.
 *
 *@author Dan Higgins NCEAS UC Santa Barbara
 *
 */

public class ZipFiles extends TypedAtomicActor
{
 /**
  * An array of file names of the files to be put into a zipped files
  */
  public TypedIOPort zipFilenamesArray = new TypedIOPort(this, "zipFilenamesArray", true, false);

 /**
  * The name of the zip file to be created
  */
  public TypedIOPort zippedFileName = new TypedIOPort(this, "zippedFileName", true, false);

 /**
  * The name of the zip file that was created; acts as an output trigger
  */
  public TypedIOPort zippedFileResult = new TypedIOPort(this, "zippedFileResult", false, true);
  


  public ZipFiles(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException
  {
    super(container, name);
    
    
    zipFilenamesArray.setTypeEquals(new ArrayType(BaseType.STRING));
    zippedFileName.setTypeEquals(BaseType.STRING);
    zippedFileResult.setTypeEquals(BaseType.STRING);

            _attachText("_iconDescription",
                "<svg>\n"
                + "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"30\" "
                + "style=\"fill:white\"/>\n"
                + "<text x=\"16\" y=\"22\" "
                + "style=\"font-size:20; fill:blue; font-family:SansSerif\">"
                + "ZIP</text>\n"
                + "</svg>\n");
    
  }
  

  /**
   *
   */
  public boolean prefire() throws IllegalActionException
  {
    return super.prefire();
  }

  /**
   *
   */
  public void fire() throws IllegalActionException
  {
    String zipNameStr = "";
    FileOutputStream fos;
    super.fire();
    try{
      if (zippedFileName.getWidth()>0) {
        zipNameStr = ((StringToken)zippedFileName.get(0)).stringValue();
//System.out.println("ZipFileName: "+zipNameStr);
        File zipOutFile = new File(zipNameStr);
        fos = new FileOutputStream(zipOutFile);
//System.out.println("FileOutputStream: "+fos);
      } else {
        System.out.println("No Zipped file name!");
        return;
      }
      if (zipFilenamesArray.getWidth()>0) {
        ArrayToken token = (ArrayToken)zipFilenamesArray.get(0);
        ZipOutputStream zipoutputstream = new ZipOutputStream(fos);
       // Select your choice of stored (not compressed) or
       //   deflated (compressed).
        zipoutputstream.setMethod(ZipOutputStream.DEFLATED);
      
      
        // now iterate over all the filenames in the array
        for (int i=0;i<token.length();i++) {
          StringToken s_token = (StringToken)token.getElement(i);
          String ascfilename = s_token.stringValue();
          File file = new File(ascfilename);
//System.out.println("File being zipped: "+file);
          byte [] rgb = new byte [1000];
          int n;
          FileInputStream fileinputstream;
        // Calculate the CRC-32 value.  This isn't strictly necessary
        //   for deflated entries, but it doesn't hurt.

          CRC32 crc32 = new CRC32();

          fileinputstream = new FileInputStream(file);

          while ((n = fileinputstream.read(rgb)) > -1) {
          crc32.update(rgb, 0, n);
          }
          fileinputstream.close();
          // Create a zip entry.
          ZipEntry zipentry = new ZipEntry(""+file);
          zipentry.setSize(file.length());
          zipentry.setTime(file.lastModified());
          zipentry.setCrc(crc32.getValue());
          // Add the zip entry and associated data.
          zipoutputstream.putNextEntry(zipentry);
          fileinputstream = new FileInputStream(file);

          while ((n = fileinputstream.read(rgb)) > -1) {
            zipoutputstream.write(rgb, 0, n);
          }

          fileinputstream.close();
          zipoutputstream.closeEntry();
          
        }
        zipoutputstream.close();
        fos.close();
      }
    } catch (Exception w) {}

        
    zippedFileResult.broadcast(new StringToken(zipNameStr));
  }
 
  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire() throws IllegalActionException 
  {
    return super.postfire();
  }

 
 }
